#pragma once
#include "Image.h"
#include "Core/GcArray.h"

namespace storm {
	STORM_PKG(graphics);

	/**
	 * A collection of images of different sizes that have the same motive. This allows providing a
	 * set of hand-tweaked images of the same motive (perhaps with different levels of details) so
	 * that other parts of the system can pick the best size depending on what size needs to be
	 * drawn. For example, this is useful for icons to windows.
	 *
	 * The images are sorted by width and then height (images should typically be the same aspect
	 * ratio).
	 */
	class ImageSet : public Object {
		STORM_CLASS;
	public:
		// Create an empty image set.
		STORM_CTOR ImageSet();

		// Create an image set with one element in it.
		STORM_CAST_CTOR ImageSet(Image *image);

		// Copy.
		ImageSet(const ImageSet &o);

		// Deep copy.
		virtual void STORM_FN deepCopy(CloneEnv *env);

		// Empty?
		Bool STORM_FN empty() const { return !images || images->filled == 0; }

		// Any?
		Bool STORM_FN any() const { return !empty(); }

		// Get the number of images in the set.
		Nat STORM_FN count() const { return images ? Nat(images->filled) : 0; }

		// Get image number i.
		Image *STORM_FN at(Nat i) const;
		Image *STORM_FN operator [](Nat i) const { return at(i); }

		// Find the image closest to the specified dimensions.
		Image *STORM_FN closest(geometry::Size size);
		Image *STORM_FN closest(Nat width, Nat height);

		// Find the nerast larger (or equal) image to the specified dimensions.
		Image *STORM_FN atLeast(geometry::Size size);
		Image *STORM_FN atLeast(Nat width, Nat height);

		// Add a new image.
		void STORM_FN push(Image *image);
		ImageSet &STORM_FN operator <<(Image *image) { push(image); return *this; }

		// String representation.
		virtual void STORM_FN toS(StrBuf *to) const;

	private:
		// Images.
		GcArray<Image *> *images;

		// Grow the array of images.
		void grow();
	};

}
