/*
 * Copyright 2012 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gradle.api.publish.maven

import org.gradle.integtests.fixtures.CrossVersionIntegrationSpec
import org.gradle.test.fixtures.maven.MavenFileRepository

class MavenPublishCrossVersionIntegrationTest extends CrossVersionIntegrationSpec {

    final MavenFileRepository repo = new MavenFileRepository(file("maven-repo"))

    def "maven publication generated by maven-publish plugin can be consumed by previous versions of Gradle"() {
        given:
        projectPublishedUsingMavenPublishPlugin()

        expect:
        canConsumePublicationWithPreviousVersion()
    }

    def projectPublishedUsingMavenPublishPlugin() {
        settingsFile.text = "rootProject.name = 'published'"

        buildFile.text = """
apply plugin: 'java'
apply plugin: 'maven-publish'

group = 'org.gradle.crossversion'
version = '1.9'

repositories {
    mavenCentral()
}
dependencies {
    compile "commons-collections:commons-collections:3.0"
}
publishing {
    repositories {
        maven { url "${repo.uri}" }
    }
}
"""

        version current withTasks 'publish' run()
    }

    def canConsumePublicationWithPreviousVersion() {
        settingsFile.text = "rootProject.name = 'consumer'"

        buildFile.text = """
configurations {
    lib
}
repositories {
    mavenCentral()
    mavenRepo(urls: ['${repo.uri}'])
}
dependencies {
    lib 'org.gradle.crossversion:published:1.9'
}
task retrieve(type: Copy) {
    into 'build/resolved'
    from configurations.lib
}
"""

        version previous withDeprecationChecksDisabled() withTasks 'retrieve' run()

        file('build/resolved').assertHasDescendants('published-1.9.jar', 'commons-collections-3.0.jar')
    }
}
