/* Copyright 2023 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "tensorflow/dtensor/mlir/expansions/where_spmd_expander.h"

#include <string>
#include <vector>

#include "llvm/ADT/DenseMap.h"
#include "mlir/IR/BuiltinAttributes.h"  // from @llvm-project
#include "mlir/IR/Operation.h"  // from @llvm-project
#include "mlir/IR/Value.h"  // from @llvm-project
#include "tensorflow/dtensor/cc/dstatus.h"
#include "tensorflow/dtensor/cc/tensor_layout.h"

namespace tensorflow {
namespace dtensor {

StatusOr<mlir::Operation*> WhereOpSPMDExpander::ExpandOp(mlir::Operation* op) {
  // Where op does not do anything during SPMD expansion.
  // Because the output layout is parted layout and it follows local semantic.
  return op;
}

StatusOr<llvm::DenseMap<int, Layout>> WhereOpSPMDExpander::ComputeLayoutForward(
    mlir::Operation* op, const llvm::DenseMap<int, Layout>& input_layouts) {
  if (input_layouts.find(0) == input_layouts.end()) {
    return llvm::DenseMap<int, Layout>();
  }
  // Currently Where op only supports 1D input.
  Layout layout = input_layouts.lookup(0);
  if (layout.rank() != 1) {
    return llvm::DenseMap<int, Layout>();
  }

  // Append an unsharded sharding spec for the index dimension generated by the
  // Where op.
  std::vector<std::string> layout_specs;
  layout_specs.push_back(layout.sharding_spec(0));
  layout_specs.push_back(Layout::kUnshardedDim);
  // The output of Where Op contains dynamic shape and has parted layout.
  // This is the source of the parted layout and it is propagated to descendent
  // ops.
  TF_ASSIGN_OR_RETURN(Layout new_layout,
                      Layout::GetLayout(Layout::LayoutType::kParted,
                                        layout_specs, layout.mesh()));
  return llvm::DenseMap<int, Layout>({{0, new_layout}});
}

StatusOr<llvm::DenseMap<int, Layout>>
WhereOpSPMDExpander::ComputeLayoutBackward(
    mlir::Operation* op, const llvm::DenseMap<int, Layout>& output_layouts) {
  if (output_layouts.find(0) == output_layouts.end()) {
    return llvm::DenseMap<int, Layout>();
  }

  // Remove the unsharded sharding spec generated by the Where op.
  Layout layout = output_layouts.lookup(0);
  std::vector<std::string> layout_specs;
  layout_specs.reserve(layout.rank() - 1);
  for (int i = 0; i < layout.rank() - 1; i++) {
    layout_specs.push_back(layout.sharding_spec(i));
  }
  TF_ASSIGN_OR_RETURN(Layout new_layout,
                      Layout::GetLayout(layout_specs, layout.mesh()));
  return llvm::DenseMap<int, Layout>({{0, new_layout}});
}

}  // namespace dtensor
}  // namespace tensorflow
