// SPDX-License-Identifier: BSD-3-Clause
// rpi-modcopy - Selectively copy kernel modules and their dependencies

#include "modcopy.hpp"

#include <CLI/CLI.hpp>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#include <sys/utsname.h>

namespace {

std::string get_running_kernel_version() {
    struct utsname uts = {};
    if (uname(&uts) == 0) {
        return uts.release;
    }
    return "";
}

// Read modules from a file, one per line, ignoring comments and empty lines
std::vector<std::string> read_module_file(const std::string& path) {
    std::vector<std::string> modules;

    std::istream* input = nullptr;
    std::ifstream file;

    if (path == "-") {
        input = &std::cin;
    } else {
        file.open(path);
        if (!file) {
            throw std::runtime_error("Cannot open module file: " + path);
        }
        input = &file;
    }

    std::string line;
    while (std::getline(*input, line)) {
        // Trim leading whitespace
        const size_t start = line.find_first_not_of(" \t");
        if (start == std::string::npos) {
            continue;
        }

        // Skip comments
        if (line[start] == '#') {
            continue;
        }

        // Trim trailing whitespace
        const size_t end = line.find_last_not_of(" \t\r\n");
        if (end == std::string::npos) {
            continue;
        }

        const std::string module = line.substr(start, end - start + 1);
        if (!module.empty()) {
            modules.push_back(module);
        }
    }

    return modules;
}

}  // namespace

// NOLINTNEXTLINE(bugprone-exception-escape) ; CLI11 uses exceptions for argument errors
int main(int argc, char* argv[]) {
    CLI::App app{"Selectively copy kernel modules and their dependencies"};
    app.set_version_flag("--version", "rpi-modcopy 1.0.0");

    modcopy::Options opts;

    // Positional arguments
    std::string source_str;
    std::string dest_str;
    app.add_option("source", source_str, "Source directory containing kernel modules")
        ->required()
        ->check(CLI::ExistingDirectory);
    app.add_option("dest", dest_str, "Destination root filesystem")->required();

    // Module specification options
    std::vector<std::string> module_args;
    app.add_option("-m,--module", module_args, "Module name or alias to include (repeatable)");

    std::vector<std::string> module_files;
    app.add_option("-f,--module-file", module_files,
                   "File containing module names, one per line (use '-' for stdin, repeatable)");

    // Kernel version
    std::string kernel_version;
    app.add_option("-k,--kernel-version", kernel_version,
                   "Kernel version string (default: running kernel)");

    // Module directory
    std::string module_dir_str = "/usr/lib/modules";
    app.add_option("-M,--module-dir", module_dir_str,
                   "Module directory path relative to source (default: /lib/modules)");

    // Behaviour flags
    app.add_flag("-v,--verbose", opts.verbose, "Produce verbose output");
    app.add_flag("-n,--dry-run", opts.dry_run,
                 "Show what would be copied without actually copying");
    app.add_flag("--keep-going", opts.keep_going,
                 "Continue processing even if some modules cannot be found");

    CLI11_PARSE(app, argc, argv);

    // Handle dry-run implying verbose
    if (opts.dry_run) {
        opts.verbose = true;
    }

    // Set paths
    opts.source = source_str;
    opts.dest = dest_str;
    opts.module_dir = module_dir_str;

    // Determine kernel version
    if (kernel_version.empty()) {
        kernel_version = get_running_kernel_version();
        if (kernel_version.empty()) {
            std::cerr << "error: could not determine running kernel version\n";
            return 2;
        }
        if (opts.verbose) {
            std::cerr << "Using running kernel version: " << kernel_version << "\n";
        }
    }
    opts.kernel_version = kernel_version;

    // Collect modules from --module arguments
    for (const auto& m : module_args) {
        opts.modules.push_back(m);
    }

    // Collect modules from --module-file arguments
    for (const auto& file : module_files) {
        try {
            auto file_modules = read_module_file(file);
            opts.modules.insert(opts.modules.end(), file_modules.begin(), file_modules.end());
        } catch (const std::exception& e) {
            std::cerr << "error: " << e.what() << "\n";
            return 2;
        }
    }

    // Check we have at least one module
    if (opts.modules.empty()) {
        std::cerr << "error: no modules specified (use --module or --module-file)\n";
        return 5;
    }

    if (opts.verbose) {
        std::cerr << "Modules to resolve: " << opts.modules.size() << "\n";
    }

    // Phase 1: Resolve modules and dependencies
    auto result = modcopy::resolve_modules(opts);
    if (result.exit_code != 0 && result.exit_code != 1) {
        return result.exit_code;
    }

    // Phase 2: Copy files
    result = modcopy::copy_modules(opts, result);
    if (result.exit_code != 0 && result.exit_code != 1) {
        return result.exit_code;
    }

    // Phase 3: Run depmod on destination
    if (!opts.dry_run) {
        const int depmod_ret = modcopy::run_depmod(opts);
        if (depmod_ret != 0) {
            return depmod_ret;
        }
    }

    // Report summary
    if (opts.verbose || result.modules_skipped > 0) {
        std::cerr << "Copied " << result.files_to_copy.size() << " files";
        if (result.modules_skipped > 0) {
            std::cerr << ", skipped " << result.modules_skipped << " modules";
        }
        std::cerr << "\n";
    }

    return result.exit_code;
}
