# rpi-modcopy

Selectively copy kernel modules and their dependencies from a source directory into a destination root filesystem. Useful for building minimal Linux images that include only the kernel modules actually required.

## Features

- Recursive dependency resolution for kernel modules
- Support for module aliases (e.g., `crypto-xts(aes)`)
- Safe handling of built-in modules
- Reads module lists from files or stdin
- Dry-run mode for previewing operations

## Building

### Debian Package (Recommended)

Install build dependencies:

```bash
sudo mk-build-deps --install --remove
```

Build the Debian package:

```bash
debuild -b -uc -us
```

The package will be created in the parent directory.

### Manual Build with Meson

```bash
meson setup builddir
meson compile -C builddir
```

The executable will be at `builddir/rpi-modcopy`.

## Development

### Code Formatting

Format all source files:

```bash
meson compile -C builddir format
```

Check formatting without modifying files:

```bash
meson compile -C builddir format-check
```

### Static Analysis

Run clang-tidy:

```bash
meson compile -C builddir tidy
```

Run cppcheck:

```bash
meson compile -C builddir cppcheck
```

Run flawfinder (security-focused analysis):

```bash
meson compile -C builddir flawfinder
```

Run include-what-you-use:

```bash
meson compile -C builddir iwyu
```

### Sanitizer Builds

Helper scripts are provided in `scripts/` to configure sanitizer builds:

**AddressSanitizer (memory errors, leaks):**

```bash
./scripts/configure-asan.sh
meson compile -C builddir-asan
ASAN_OPTIONS=detect_leaks=1 ./builddir-asan/rpi-modcopy ...
```

**UndefinedBehaviorSanitizer:**

```bash
./scripts/configure-ubsan.sh
meson compile -C builddir-ubsan
UBSAN_OPTIONS=print_stacktrace=1 ./builddir-ubsan/rpi-modcopy ...
```

**ThreadSanitizer (data races):**

```bash
./scripts/configure-tsan.sh
meson compile -C builddir-tsan
./builddir-tsan/rpi-modcopy ...
```

**MemorySanitizer (uninitialised reads):**

```bash
./scripts/configure-msan.sh
meson compile -C builddir-msan
./builddir-msan/rpi-modcopy ...
```

Note: MSAN requires all linked libraries to be instrumented; expect false positives when linking against system libraries.

### Suppressing Lint Warnings

When suppressing a lint warning, always include a justification:

**clang-tidy:**

```cpp
// NOLINTNEXTLINE(check-name) ; reason for suppression
code_here();

code_here();  // NOLINT(check-name) ; reason for suppression
```

**cppcheck:**

```cpp
// cppcheck-suppress checkName ; reason for suppression
code_here();
```

## Usage

```bash
rpi-modcopy <source> <dest> --module <name> [--module <name>...]
rpi-modcopy <source> <dest> --module-file <file>
```

See `man rpi-modcopy` for full documentation.

## Examples

Copy specific modules to a target rootfs:

```bash
rpi-modcopy /usr/lib/modules /mnt/rootfs \
    --module configs \
    --module usbfunc:ffs
```

Copy modules listed in a file:

```bash
rpi-modcopy /usr/lib/modules /mnt/rootfs \
    --module-file modules.list
```

## Licence

BSD-3-Clause
