/**
 * Copyright (c) 2019-2025 Hailo Technologies Ltd. All rights reserved.
 * Distributed under the MIT license (https://opensource.org/licenses/MIT)
 **/
/**
 * @file firmware_status.h
 * @brief Defines firmware status codes.
**/

#ifndef __FIRMWARE_STATUS_H__
#define __FIRMWARE_STATUS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "status.h"
#include "utils.h"


#define FIRMWARE_STATUS__MASK_FROM_OFFSET_AND_WIDTH(__offset, __width) (((1UL << (__width)) -1UL) << (__offset))
#define FIRMWARE_STATUS__FIELD_EXTRACT(__input, __offset, __width) \
    (((FIRMWARE_STATUS__MASK_FROM_OFFSET_AND_WIDTH((__offset), (__width)) & (__input)) >> (__offset)))

/*
 * +--------------+------------+------------+------------+
 * |  Bits 0-15   | Bits 16-23 | Bits 24-27 | Bits 28-31 |
 * +--------------+------------+------------+------------+
 * | Status value | Module     | Reserved   | Component  |
 * +--------------+------------+------------+------------+
 *
 * */

#define FIRMWARE_STATUS__COMPONENT_OFFSET_BITS (28UL)
#define FIRMWARE_STATUS__COMPONENT_WIDTH_BITS (4UL)
#define FIRMWARE_STATUS__COMPONENT_MASK (FIRMWARE_STATUS__MASK_FROM_OFFSET_AND_WIDTH( \
            FIRMWARE_STATUS__COMPONENT_OFFSET_BITS, FIRMWARE_STATUS__COMPONENT_WIDTH_BITS))
#define FIRMWARE_STATUS__COMPONENT_MAX ((1UL << FIRMWARE_STATUS__COMPONENT_WIDTH_BITS) - 1UL)
#define FIRMWARE_STATUS__COMPONENT_SHIFT(__component) ((__component) << FIRMWARE_STATUS__COMPONENT_OFFSET_BITS)
#define FIRMWARE_STATUS__COMPONENT_GET(__status) (FIRMWARE_STATUS__FIELD_EXTRACT((__status), \
            FIRMWARE_STATUS__COMPONENT_OFFSET_BITS, FIRMWARE_STATUS__COMPONENT_WIDTH_BITS))

#define FIRMWARE_STATUS__MODULE_INDEX_OFFSET_BITS (16UL)
#define FIRMWARE_STATUS__MODULE_INDEX_WIDTH_BITS (8UL)
#define FIRMWARE_STATUS__MODULE_INDEX_MASK (FIRMWARE_STATUS__MASK_FROM_OFFSET_AND_WIDTH( \
            FIRMWARE_STATUS__MODULE_INDEX_OFFSET_BITS, FIRMWARE_STATUS__MODULE_INDEX_WIDTH_BITS))
#define FIRMWARE_STATUS__MODULE_INDEX_MAX ((1UL << FIRMWARE_STATUS__MODULE_INDEX_WIDTH_BITS) - 1UL)
#define FIRMWARE_STATUS__MODULE_INDEX_SHIFT(__module) ((__module) << FIRMWARE_STATUS__MODULE_INDEX_OFFSET_BITS)
#define FIRMWARE_STATUS__MODULE_INDEX_GET(__status) (FIRMWARE_STATUS__FIELD_EXTRACT((__status), \
            FIRMWARE_STATUS__MODULE_INDEX_OFFSET_BITS, FIRMWARE_STATUS__MODULE_INDEX_WIDTH_BITS))

#define FIRMWARE_STATUS__VALUE_OFFSET_BITS (0UL)
#define FIRMWARE_STATUS__VALUE_WIDTH_BITS (16UL)
#define FIRMWARE_STATUS__VALUE_MASK (FIRMWARE_STATUS__MASK_FROM_OFFSET_AND_WIDTH( \
            FIRMWARE_STATUS__VALUE_OFFSET_BITS, FIRMWARE_STATUS__VALUE_WIDTH_BITS))
#define FIRMWARE_STATUS__VALUE_MAX ((1UL << FIRMWARE_STATUS__VALUE_WIDTH_BITS) - 1UL)
#define FIRMWARE_STATUS__VALUE_SHIFT(__value) ((__value) << FIRMWARE_STATUS__VALUE_OFFSET_BITS)
#define FIRMWARE_STATUS__VALUE_GET(__status) (FIRMWARE_STATUS__FIELD_EXTRACT((__status), \
            FIRMWARE_STATUS__VALUE_OFFSET_BITS, FIRMWARE_STATUS__VALUE_WIDTH_BITS))

#define FIRMWARE_STATUS__COMPONENT_ID (4)


/* 
Updating rules:
1. Always add new statues at the end of the module's statuses
2. Always add new modules at the end
3. DO NOT remove existing statuses
4. DO NOT remove existing modules
5. DO NOT change the order of existing statuses or modules
*/

#define FIRMWARE_STATUS__VARIABLES \
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__GENERAL)\
   FIRMWARE_STATUS__X(HAILO_STATUS_UNINITIALIZED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_INVALID_INTERRUPT_HANDLER)\
   FIRMWARE_STATUS__X(HAILO_STATUS_CONVERT_TO_TEXTUAL_FAILED)\
   FIRMWARE_STATUS__X(SYSTEM_STATUS_STATUS_INVALID_PARAMS)\
   FIRMWARE_STATUS__X(HAILO_STATUS_UNREACHABLE)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_UNDEFINED_EXCEPTION)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_NMI)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_HARD_FAULT)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_MPU_FAULT)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_BUS_FAULT)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_USAGE_FAULT)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_SMART_FAIL)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_SMART_TIMEOUT)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_SMART_NOT_STARTED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_SMART_UNSUPPORTED_OPERATION)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_SMART_BIST_NULL_PARAMS)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_SMART_BIST_INVALID_BYPASS)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SOC_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_QSPI_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_INITIAL_I2C_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SOC_LOAD_CLOCK_FROM_CONFIG_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SECOND_I2C_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SETTING_PCIE_POWER_MODES_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ETHERNET_SERVICE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_LWIP_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_CONTROL_TASK_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_PCIE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SECURE_SERVICE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_MAILBOX_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SENDING_CORE_CPU_LOADED_EVENT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_INTER_CPU_EVENT_MANAGER_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SOC_CORE_CPU_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_UPDATE_CLOCK_CORE_CPU_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_CONTROL_ETH_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_SEMAPHORE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_DRAM_DMA_SERVICE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_VDMA_SERVICE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_STATUS_ERROR_HANDLING_STACK_OVERFLOW)\
   FIRMWARE_STATUS__X(HAILO_STATUS_UNEXPECTED_NULL_ARGUMENT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__DATAFLOW)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_PARAMETER)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_STREAM_INDEX)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_STREAM_ALREADY_USED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INBOUND_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_OUTBOUND_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_SET_QUEUE_UDP_FILTER_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_RECEIVE_COMMUNICATION)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_QUEUE_UDP_DATAFLOW_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_FREE_TX_DESCIPTORS_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_ETHERNET_DATAFLOW_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_ETHERNET_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_STREAM_ALREADY_CLOSED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_BYTES_PER_BUFFER_VALUE)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_COMMUNICATION_TYPE)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_STREAM_NOT_CONFIGURED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_FIFO_ADDRESS_NOT_FOUND)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_GET_QUEUE_TX_MEMORY_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_BUFFERS_THRESHOLD)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_RELEASE_DATAFLOW_RX_DESCRIPTORS_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_UDP_TX_MEMORY_FULL)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_MIPI_START_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_MIPI_TX_IS_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_PACKETS_PER_SYNC_OVERFLOWS)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_PCIE_CONFIGURE_CHANNEL_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_PCIE_RELEASE_CHANNEL_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_UNSUPPORTED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_ALLOCATE_RX_QUEUE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_UNUSED_DATAFLOW_MANAGER)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_LATENCY_MEASUREMENT_IN_PROGRESS)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_LATENCY_MEASUREMENT_OVERFLOW)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_LATENCY_MEASUREMENT_UNDERFLOW)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_LATENCY_MEASUREMENT_READ_WHILE_DISABLED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INPUT_SYNC_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_NO_AVAILABLE_DATAFLOW_MANAGER)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_INVALID_DATAFLOW_MANAGER_ID)\
   FIRMWARE_STATUS__X(HAILO_DATAFLOW_STATUS_DATAFLOW_IS_ACTIVE_ON_ANOTHER_MANAGER)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__ETHERNET_SERVICE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_INVALID_PARAMS)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_CEDI_PROBE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MISMATCHING_PDATA_SIZE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MISMATCHING_RX_DESC_LIST_SIZE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MISMATCHING_TX_DESC_LIST_SIZE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MISMATCHING_STATISTICS_SIZE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_ADD_RX_BUFFER_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_ADD_DATAFLOW_RX_PAYLOAD_BUFFER_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_ADD_DATAFLOW_RX_HEADER_BUFFER_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_CEDI_INIT_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_READ_RX_BUFFER_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_READ_RX_HEADER_BUFFER_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_READ_RX_PAYLOAD_BUFFER_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_HEADER_NO_DATA)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_PAYLOAD_NO_DATA)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_HEADER_UNEXPECTED_FRAME_DESCRIPTOR)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_PAYLOAD_UNEXPECTED_FRAME_DESCRIPTOR)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_TX_QUEUE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_FREE_TX_DESCRIPTOR_ERROR)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_T1_SCREEN_REG_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_DATAFLOW_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_DATAFLOW_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MDIO_WRITE_TIMEOUT)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MDIO_WRITE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MDIO_READ_TIMEOUT)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MDIO_READ_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_QN_EVENT_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_DATAFLOW_EVENT_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_TX_QUEUE_SEG_ALLOC_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_TX_CHECKSUM_OFFLOAD_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_HDR_DATA_SPLIT_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_INVALID_MAC_SIZE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_GET_SPECIFIC_ADDR_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_FREE_PAYLOAD_TX_DESCRIPTOR_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_FREE_HEADER_TX_DESCRIPTOR_ERROR)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MISMATCHING_FIFO_ADDRESS)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_OPTIMIZED_EDD_DATAFLOW_TX_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_Q0_EVENT_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SWAP_DATAFLOW_DESCRIPTOR_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_DATAFLOW_RX_NO_DATA)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_UNALIGNED_RX_INJECTION)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_DATAFLOW_HANDLE_INJECTION_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_UNEXPECTED_FRAME_SIZE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_NO_DATA)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_UNEXPECTED_FRAME_DESCRIPTOR)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_SET_TIMER_INC_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_DATAFLOW_RX_QUEUE_UNAVAILABLE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_DATAFLOW_RX_QUEUE_ALREADY_FREE)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RESET_RX_QUEUE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_RX_STREAM_TO_ETHERNET_QUEUE_FAILED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_ETHERNET_UNSUPPORTED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_MDIO_UNSUPPORTED)\
   FIRMWARE_STATUS__X(ETHERNET_SERVICE_STATUS_PHY_RESET_TIMEOUT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__CONTROL)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_PCIE_CONTROL_MODE)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_MESSAGE_QUEUE_CREATE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_VERSION)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_OVERRUN_BEFORE_PARAMETERS)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_OVERRUN_AT_PARAMETER)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_CONTROL_ACK_RECEIVED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_OPCODE)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_COMMUNICATION_TYPE)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_LWIP_TX_QUEUE_FRAME_FAILED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_PCIE_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_PCIE_REPLY_FAILED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INIT_RESPONSE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_DEVICE_ARCHITECTURE)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_PLATFORM_ID)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_UNSUPPORTED_OPCODE)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_INVALID_PERCENTAGE_RECEIVED)\
   FIRMWARE_STATUS__X(HAILO_CONTROL_STATUS_COMMUNICATION_PARAMS_UNINITIALIZED)\
   \
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INCORRECT_PARAMETER_COUNT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_ADDRESS_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_DATA_COUNT_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_STREAM_INDEX_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_IS_INPUT_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_COMMUNICATION_TYPE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_COMMUNICATION_TYPE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_GET_ETHERNET_HEADER_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_COMMUNICATION_PARAMS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_CONFIG_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_OPEN_STREAM_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_CLOSE_STREAM_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_INVALID_PACKETS_INPUT_BUFFERS_RATIO)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_OPERATION_TYPE_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_OPERATION_TYPE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_RESET_PHY_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_RESET_TYPE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_SYSTEM_RESET_RETURNED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_NOT_IDLE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BYTES_PER_BUFFER_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BUFFERS_PER_FRAME_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_SET_STREAM_REGISTERS_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_STREAM_INDEX)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONFIG_TYPE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONFIG_CORE_TOP_TYPE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONFIG_CORE_TOP_CONFIG_PARAMS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_STREAM_THRESHOLD_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_POWER_MEASUREMENT_READ)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_SET_POWER_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_GET_POWER_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_START_POWER_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_STOP_POWER_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_SINGLE_POWER_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_POWER_MEASUREMENT_INDEX)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_UPDATE_START_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_REGISTER_ADDRESS_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_WRITE_FIRMWARE_UPDATE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_VALIDATE_FIRMWARE_UPDATE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_MD5_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_FIRMWARE_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FINISH_FIRMWARE_UPDATE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_USER_CONFIG_EXAMINE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_USER_CONFIG_ERASE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_I2C_WRITE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_I2C_READ_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_IS_FIRST_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_DATA_SIZE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_START_OFFSET_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_RESET_DATA_SIZE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SENSOR_TYPE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SECTION_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_TOTAL_DATA_SIZE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SLAVE_ADDRESS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SLAVE_REGISTERS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SLAVE_OFFSET_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SLAVE_BUS_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SLAVE_BUS_HOLD_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SENSOR_CONFIG_SLAVE_ENDIANNESS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_LATENCY_CONFIG_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_LATENCY_READ_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_DATA_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_MIPI_INIT_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_PARSE_COMMAND_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_EXECUTE_COMMAND_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_INVALID_FRAMES_PER_SYNC)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_INVALID_PACKETS_PER_FRAME)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DATAFLOW_INVALID_SYNC_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_PARSE_MESSAGE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_PLATFORM_ID)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_DEVICE_ARCHITECTURE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_UNSUPPORTED_OPCODE_FOR_DEVICE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_USE_RTP_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_DATA_RATE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INTER_CPU_REQUEST_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_INTER_CPU_OPCODE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CORE_CPU_RETURNED_STATUS_FAILURE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_VERSION_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_APP_COUNT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_APP_HEADER_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_CLUSTER_END_GARANTEE_TRIGGER_LENGTH)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_NUMBER_OF_EDGE_LAYERS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_NUMBER_OF_ACTION_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_TRIGGER_GROUPS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CONTEXT_SWITCH_CONTEXT_NETWORK_DATA_LENGTH_HIGHER_THEN_MAX_CONTROL_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_IDLE_TIME_MEASUREMENT_ALREADY_SET)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_IDLE_TIME_GET_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_IDLE_TIME_SET_MEASUREMENT)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_CONTEXT_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_PCIE_CONFIG_CHANNEL_NUMBER_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_STATE_MACHINE_STATUS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(CONTROL_TASK_STATUS_ACTION_LIST_LENGTH_EXITED_BUFFER_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CPU_ID_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SHOULD_ENABLE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_WD_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_WD_CYCLES_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_WD_CONFIG_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_PREVIOUS_SYSTEM_STATE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CPU_ID)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SLICING_APPROACH_LENGTH) /* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_INTERRUPT_TYPE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_INTERRUPT_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_INTERRUPT_SUB_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_D2H_EVENT_MANAGER_SET_NEW_HOST_INFO_FAIL)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_D2H_EVENT_MANAGER_SET_NEW_HOST_INFO_INVALID_CONNECTION_TYPE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_D2H_EVENT_MANAGER_SET_NEW_HOST_INFO_INVALID_IP_ADDR_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_D2H_EVENT_MANAGER_SET_NEW_HOST_INFO_INVALID_PORT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_D2H_EVENT_MANAGER_HOST_INFO_INVALID_PRIORITY_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_D2H_EVENT_MANAGER_HOST_INFO_SEND_EVENT_FAIL)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CANT_GET_PCIE_CONTROL_PACKET)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_SENDING_DATA_TO_CONTROL_QUEUE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_GOT_INVALID_DVM_FOR_THAT_PLATFORM)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_APPLICATION_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CANT_GET_CHIP_TEMPERATURE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_DIRECT_EEPROM_ADDRESS_ACCESS_NOT_ALLOWED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_ACTION_LIST_OFFSET_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CANT_GET_SOC_ID)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CANT_GET_LCS)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_GET_ETH_MAC_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_ENABLE_DEBUGGING_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SKIP_NN_STREAM_CONFIG_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_APPLICATION_BATCH_SIZE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_MIPI_INCORRECT_IMG_IN_DATA_TYPE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_MIPI_INCORRECT_PIXELS_PER_CLOCK)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BREAKPOINT_ID_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BREAKPOINT_CONTROL_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BREAKPOINT_DATA_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_IS_FIRST_CONTROL_PER_CONTEXT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_IS_LAST_CONTROL_PER_CONTEXT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_IS_RMA_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_PCIE_DATAFLOW_TYPE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_PCIE_CONFIG_CHANNEL_NUMBER)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_STREAM_REMAP_DATA_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CONTEXT_SWITCH_UNSUPPORTED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_ACCESS_RESTRICTED_MEMORY_AREA)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_REQUSET_LENGTH_OVERFLOW)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_REQUSET_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_UNEXPECTED_ACK_VALUE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_PART_OF_THE_MESSAGE_NOT_PARSED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_CONTEXT_CFG_BASE_ADDRESS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_CONTEXT_CFG_TOTAL_DESCRIPTORS_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_ROW_SIZE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_POWER_MODE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SECOND_STAGE_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_COPY_SECOND_STAGE_TO_FLASH_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_MIPI_MISMATCH_SIZES_BETWEEN_MIPI_OUTPUT_AND_STREAM)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_MIPI_MISMATCH_SIZES_BETWEEN_OUTPUT_DATA_TYPE_AND_PIXELS_PER_CLOCK)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_VALIDATION_FEATURES_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_OBSOLETE_CONTROL)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BATCH_INDEX_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_ENABLE_USER_CONFIGURATION_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_SYSTEM_FORCED_RESET_RETURNED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_LOGGER_LEVEL)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_LOGGER_INTERFACE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_IS_TOP_TEST_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_TOP_BYPASS_BITMAP_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CLUSTER_BITMAP_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CLUSTER_BYPASS_BITMAP_0_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CLUSTER_BYPASS_BITMAP_1_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_BIST_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BYPASS_SELECTION)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_ONLY_POWER_CAN_BE_SAMPLED_WITH_EVB_TOTAL_POWER_OPTION)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_UNSUPPORTED_DEVICE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_DATAFLOW_MANAGER_ID_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_NN_STREAM_CONFIG_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_IS_SINGLE_CONTEXT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SLAVE_ENDIANNESS_VALUE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_PVT_IS_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_SETTING_THROTTLING_STATE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_FAILED_SETTING_OVERCURRENT_STATE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CONTROL_UNSUPPORTED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_CONTROL_DEPRECATED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_HOST_BUFFER_INFO)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CFG_CHANNELS_COUNT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_DYNAMIC_BATCH_SIZE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_INFER_FEATURES_LENGTH) /* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONFIG_CHANNEL_INFOS)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_IS_BATCH_SIZE_FLOW_LENGTH) /* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_CONTEXT_TYPE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CONTEXT_SWITCH_CONTEXT_NETWORK_GROUP_ID_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_SET_SLEEP_STATE_FAILED)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SLEEP_STATE_SIZE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_SLEEP_STATE)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_HW_INFER_STATE_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CHANNELS_INFO_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BATCH_COUNT_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_CACHE_INFO_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_READ_OFFSET_DELTA_LENGTH)\
   FIRMWARE_STATUS__X(CONTROL_PROTOCOL_STATUS_INVALID_BOUNDARY_CHANNELS_MODE_LENGTH)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__POWER_MEASUREMENT)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_POWER_INIT_ERROR)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_DVM_ERROR)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_INVALID_GLOBAL_ARRAY)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_MAIN_TASK_RETURNED)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_EVENT_BIT_NOT_SET)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_EVENT_BIT_NOT_CLEARED)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_INVALID_MEASUREMENT_ARGUMENTS)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_INVALID_INTERVAL)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_ALREADY_STARTED)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_INIT_DVMS_FAILED)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_STATUS_INVALID_DVM)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_TASK_NOT_RUNNING)\
   FIRMWARE_STATUS__X(HAILO_POWER_MEASUREMENT_DVM_ALREADY_IN_USE_FOR_PERIODIC_SAMPLING)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__MIPI)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_RX_ID)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_TX_ID)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_UNINITIALIZED_CONFIG)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_NULL_CONFIG_PASSED)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_TIMEOUT)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_NUMBER_OF_LANES)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_DATA_RATE)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_TX_DATA_TYPE)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_VIRTUAL_CHANNEL)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_CONFIG_CAMERA_FAILED)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INCORRECT_CAMERA_CONFIG_INDEX)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_RESET_CAMERA_FAILED)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INVALID_PARAMS)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_DATAFLOW_PIPE_UNAVAILABLE)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_PIPE_INDEX_DIDNT_INITIALIZED)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_DPHY_PIPE_DIDNT_SET)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_DPHY_PIPE_DIDNT_ALLOCATED)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_MIPI_RX_ID_2_VALID_ONLY_IN_B0) /* DEPRECATED */\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_MIPI_RX_PHY_WRITE_FAIL)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_MIPI_UNSUPPORTED)\
   FIRMWARE_STATUS__X(HAILO_MIPI_STATUS_INCORRECT_ISP_WINDOW_INPUT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__USER_CONFIG)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_INVALID_PARAMS)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_INVALID_MAGIC)\
   FIRMWARE_STATUS__X(USER_CONFIG_INVALID_VERSION)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_INVALID_CATEGORY)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_INVALID_ENTRY_ID)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_INVALID_ENTRY_SIZE)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_MAXIMUM_SIZE_EXCEEDED)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_SECTION_ERASE_FAILED)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_READ_FROM_EEPROM_FAILED)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_READ_FROM_FLASH_FAILED)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_READ_FROM_PCIE_PRELOAD_FAILED)\
   FIRMWARE_STATUS__X(USER_CONFIG_STATUS_NULL_POINTER_PASSED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__I2C)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_UNSUPPORTED_DEVICE)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_VENDOR_INIT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_VENDOR_START_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_VENDOR_READ_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_VENDOR_WRITE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_VENDOR_TIMEOUT_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_MISMATCHING_PDATA_SIZE)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INVALID_WRITE_SIZE)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INVALID_SLAVE_INDEX)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INVALID_BUS_INDEX)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_SEMAPHORE_CREATION_FAILED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_MODULE_UNINITIALIZED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_FAILED_TO_ACQUIRE_SEMAPHORE)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_MISMATCHING_REGS_SIZE)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INVALID_ENDIANNESS)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INVALID_OPERATION_LENGTH)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_UNSUPPORTED_BUS_SPEED)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_FAILED_TO_RELASE_SEMAPHORE)\
   FIRMWARE_STATUS__X(HAILO_I2C_STATUS_INVALID_SLAVE_ENDIANNESS_VALUE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__LWIP)\
   FIRMWARE_STATUS__X(LWIP_PORT_STATUS_INVALID_PARAMS)\
   FIRMWARE_STATUS__X(LWIP_MAIN_LOOP_STATUS_FAILED_ADDING_NETWORK_INTERFACE)\
   FIRMWARE_STATUS__X(LWIP_MAIN_LOOP_STATUS_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(LWIP_MAIN_LOOP_STATUS_DHCP_START_FAILED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_INVALID_QUEUE_NUMBER)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_QUEUE_ALREADY_USED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_SET_UDP_FILTER_FAILED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_INVALID_SOCKET_ID)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_UDP_PCB_NEW_FAILED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_UDP_BIND_FAILED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(LWIP_CONFIG_STATUS_LWIP_SEND_FAILED)\
   FIRMWARE_STATUS__X(LWIP_TX_STATUS_QUEUE_INIT_FAILED)\
   FIRMWARE_STATUS__X(LWIP_TX_STATUS_QUEUE_SEND_FAILED)\
   FIRMWARE_STATUS__X(LWIP_TX_STATUS_QUEUE_RECEIVE_FAILED)\
   FIRMWARE_STATUS__X(LWIP_TX_STATUS_QUEUE_EMPTY)\
   FIRMWARE_STATUS__X(LWIP_TX_STATUS_PBUF_ALLOC_FAILED)\
   FIRMWARE_STATUS__X(LWIP_TX_STATUS_UDP_SENDTO_FAILED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__PARALLEL_CAMERA)\
   FIRMWARE_STATUS__X(HAILO_PARALLEL_STATUS_INTERRUPT_SETUP_FAILED)\
   FIRMWARE_STATUS__X(HAILO_PARALLEL_STATUS_INTERRUPT_ENABLE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_PARALLEL_STATUS_ERROR_INDICATION)\
   FIRMWARE_STATUS__X(HAILO_PARALLEL_STATUS_I2C_CONFIG_FAILED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__QSPI)\
   FIRMWARE_STATUS__X(QSPI_STATUS_PROBE_FAILED)\
   FIRMWARE_STATUS__X(QSPI_STATUS_PRIVATE_DATA_SIZE_MISMATCH)\
   FIRMWARE_STATUS__X(QSPI_STATUS_INIT_FAILED)\
   FIRMWARE_STATUS__X(QSPI_STATUS_DISABLE_WRITE_PROTECT_FAILED)\
   FIRMWARE_STATUS__X(QSPI_STATUS_MODULE_UNITIALIZED)\
   FIRMWARE_STATUS__X(QSPI_STATUS_MISALIGNED_ADDRESS)\
   FIRMWARE_STATUS__X(QSPI_STATUS_BLOCK_ERASE_FAILED)\
   FIRMWARE_STATUS__X(QSPI_STATUS_CLEAR_AHB_REMAP_FAILED)\
   FIRMWARE_STATUS__X(QSPI_STATUS_NOT_SUPPORTED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__PCIE_SERVICE)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_PARAMETERS)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INTERRUPT_HANDLER_FAILED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INTERRUPT_CORE_HANDLER_FAILED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_MISMATCHING_MD5)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_UNSUPPORTED_SOC)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_DEBUG_INSUFFICIENT_MEMORY) /* DEPRECATED - See MEMORY_LOGGER */\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_NULL_POINTER_PASSED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_HOST_BASE_ADDRESS_IS_NOT_64KB_ALIGNED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_NOT_ENOUGH_DESCRIPTORS_IN_HOST)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_PCIE_NOT_CONFIGURED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_PCIE_UNSUPPORTED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_GOT_UNALIGNED_ADDRESS_FOR_PCIE_BUFFERS) /* DEPRECATED - See MEMORY_LOGGER */\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_DEBUG_CHIP_OFFSET) /* DEPRECATED - See MEMORY_LOGGER */\
   FIRMWARE_STATUS__X(PCIE_SERVICE__CLEARING_CREDITS_TO_OPEN_CHANNEL_IS_NOT_ALLOWED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_GLUE_LOGIC_MUST_BE_DISABLED_WHEN_CLEARING_CHANNEL_CREDITS)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_VDMA_MUST_BE_DISABLED_WHEN_CLEARING_CHANNEL_CREDITS)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_TRYING_TO_RELEASE_NON_CFG_FLOW_CHANNEL)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_H2D_CREDITS)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_D2H_CREDITS)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_DESCRIPTOR_COUNT_INDICATE_NOT_ALL_DATA_WAS_TRANSFERRED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_CHANNEL_CANNOT_FETCH_DESCRITPROTS_WHILE_ABORTED)\
   FIRMWARE_STATUS__X(PCIE_SERVICE__WAIT_UNTIL_CHANNEL_IS_IDLE_REACHED_TIMEOUT)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_UNSUPPORTED_PERIPH_BYTES_PER_BUFFER)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_GLUE_LOGIC_CHANNEL_OUT_OF_RANGE)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_H2D_GLUE_LOGIC_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_D2H_GLUE_LOGIC_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_INVALID_INITIAL_CREDIT_SIZE)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_ERROR_ADDING_CREDITS_TO_PCIE_CHANNEL)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_STREAM_INDEX)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_CHANNEL_TYPE)\
   FIRMWARE_STATUS__X(PCIE_SERVICE_STATUS_INVALID_DESC_PAGE_SIZE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__FIRMWARE_UPDATE)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_INVALID_PARAMETERS)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_ERASE_FIRMWARE_SECTION_FAILED)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_FIRMWARE_SIZE_EXCEEDED)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_FIRMWARE_UPDATE_NOT_STARTED)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_FIRMWARE_UPDATE_NOT_VALIDATED)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_UPDATE_SWITCH_FAILED)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_MISMATCHING_MD5)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_UPDATE_SWITCH_MISMATCH)\
   FIRMWARE_STATUS__X(FIRMWARE_UPDATE_STATUS_INVALID_FIRMWARE)\
   FIRMWARE_STATUS__X(FIRMWARE_SWITCH_STATUS_ERASE_SWITCH_BLOCK_FAILED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__SENSOR_CONFIG)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_INVALID)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_STORE_CONFIG_FAIL)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_LOAD_CONFIG_FAIL)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_GET_CONFIG_FAIL)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_INVALID_PARAMS)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_INVALID_SENSOR_TYPE)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_ERASE_SECTION_FAILED)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_I2C_FAILED)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_SET_I2C_SLAVE_FAILED)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_I2C_SLAVE_NOT_SET)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_GET_SECTIONS_INFO_FAILED)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_SECTION_CONFIG_NOT_SET)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_RESET_FAILED)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_SECTION_CONFIG_VERSION_MISMATCH)\
   FIRMWARE_STATUS__X(SENSOR_CONFIG_STATUS_SENSOR_SET_I2C_BUS_INDEX_FAIL)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__DEBUG)\
   FIRMWARE_STATUS__X(LOGGER_STATUS_INVALID)\
   FIRMWARE_STATUS__X(LOGGER_STATUS_INVALID_PARAM_COUNT)\
   FIRMWARE_STATUS__X(LOGGER_STATUS_QUEUE_SEND_FAILED) /* DEPRECATED - See UART_LOGGER */\
   FIRMWARE_STATUS__X(LOGGER_STATUS_FAILED_ACQUIRE_SEMAPHORE) /* DEPRECATED - See UART_LOGGER */\
   FIRMWARE_STATUS__X(LOGGER_STATUS_FAILED_RELEASE_SEMAPHORE) /* DEPRECATED - See UART_LOGGER */\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__MAILBOX)\
   FIRMWARE_STATUS__X(MAILBOX_STATUS_GOT_UNSUPPORTED_CPU_ID)\
   FIRMWARE_STATUS__X(MAILBOX_STATUS_SETUP_INTERRUPT_FAILED)\
   FIRMWARE_STATUS__X(MAILBOX_STATUS_MAILBOX_INDEX_OUT_OF_BAND)\
   FIRMWARE_STATUS__X(MAILBOX_STATUS_READ_NULL_VALUE_FROM_THE_MAILBOX)\
   FIRMWARE_STATUS__X(MAILBOX_STATUS_GOT_NULL_ARGUMENT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__SEMAPHORE)\
   FIRMWARE_STATUS__X(SEMAPHORE_STATUS_GOT_UNSUPPORTED_CPU_ID)\
   FIRMWARE_STATUS__X(SEMAPHORE_STATUS_TRY_GET_RETURN_INVALID_VALUE)\
   FIRMWARE_STATUS__X(SEMAPHORE_STATUS_INVALID_STATUS_RESPONSE)\
   FIRMWARE_STATUS__X(SEMAPHORE_STATUS_INDEX_OUT_OF_BAND)\
   FIRMWARE_STATUS__X(SEMAPHORE_STATUS_CLEAR_FAILURE)\
   FIRMWARE_STATUS__X(SEMAPHORE_STATUS_GOT_NULL_ARGUMENT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__INTER_CPU_CONTROL)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_UNSUPPORTED_OPCODE)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_RAECHED_TIMEOUT_WHILE_WAITING_FOR_RESPONSE)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_UNEXPECTED_RESPONSE_OPCODE)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_UNSUPPORTED_CONTROL_PROTOCOL_VERSION)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_SEQUENCE_RETRANSMITTED)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_APP_CPU_REACHED_TIMEOUT)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_UNEXPECTED_SEQUENCE_NUMBER)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_INVALID_PARAMETERS)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_INVALID_REQUEST_SIZE)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_INVALID_RESPONSE_SIZE)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_RECEIVED_NULL_VALUE_FROM_MAILBOX_READ_FUNCTION)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__INTER_CPU_EVENT_MANAGER)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_FAILED_INITIALIZE_SEMAPHORE)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_EVENT_INDEX_OUT_OF_RANGE)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_TRYING_TO_CLEAR_EVENT_THAT_DIDNT_HAPPEN)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_GOT_NULL_POINTER_TO_EVENT)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_GOT_UNVALID_EVENT)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_WAIT_FOR_EVENT_CLEAR_REACHED_TIMEOUT)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_WAIT_FOR_EVENT_REACHED_TIMEOUT)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_FAILED_TO_ACQUIRE_SEMAPHORE)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_GOT_NULL_ARGUMENT_FROM_MAILBOX_CHECK_DATA)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_GOT_UNSUPPORTED_CPU_ID)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_PAYLOAD_SIZE_IS_HIGHER_THEN_MAX_SIZE)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_GOT_NULL_POINTER_TO_PAYLOAD)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_EVENT_WAS_OVERWRITTEN)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_INVALID_ARGS)\
   FIRMWARE_STATUS__X(INTER_CPU_EVENT_MANAGER_STATUS_GOT_EVENT_INDICATION_BUT_CHECK_EVENT_RETURNED_FALSE)\
   FIRMWARE_STATUS__X(INTER_CPU_CONTROL_STATUS_INVALID_CONNECTION_TYPE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__CONTEXT_SWITCH_TASK)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_RECEIVED_INVALID_VERSION)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_MAIN_HEADER_INDICATES_THERE_IS_NO_CONTEXTS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_NULL_POINTER_PASSED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_CONTEXT_HEADER_INDICATES_THERE_IS_NO_CONTEXT_NETWORK_INFO)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_MAIN_HEADER_INDICATES_THERE_IS_NO_APPLICATIONS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_USER_SENT_CONTEXT_CONTROL_BEFORE_MAIN_HEADER_CONTROL)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_CONTEXT_INFO_RECEIVED_TOO_MANY_CONTEXTS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_RECEIVED_INVALID_TRIGGER_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_READ_INVALID_SLICING_APPROACH)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_RECEIVED_INVALID_ACTION_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_UNDEFINED_MAPPING_FOR_SHIMFO)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_MISSMATCH_BETWEEN_TRIGGER_DIRECTION_AND_MAPPING_DIRECTION)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_SLICING_FUNCTION_REACHED_FORBIDDEN_MEMORY_SPACE)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_ADD_TRIGGER_FUNCTION_REACHED_FORBIDDEN_MEMORY_SPACE)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_ADD_ACTION_FUNCTION_REACHED_FORBIDDEN_MEMORY_SPACE)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_ADD_CONTEXT_INFO_FUNCTION_REACHED_FORBIDDEN_MEMORY_SPACE)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_PARSING_ERROR_WHILE_READING_TRIGGER_GROUPS)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_POINTER_REACHED_FORBIDDEN_MEMORY_SPACE)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_DOWNLOAD_ACTION_LIST_INVALID_CONTEXT_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_RECEIVED_INVALID_APPLICATION_COUNT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_RECEIVED_INVALID_TOTAL_CONTEXTS_COUNT)\
   FIRMWARE_STATUS__X(CONTEXT_SWTICH_STATUS_MISALIGNMENT_ERROR_WHILE_READING_ACTIONS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_READ_INVALID_INTERRUPT_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_READ_INVALID_ACTION_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_STATE_MACHINE_STATUS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_SLICING_APPROACH)/* DEPRECATED */\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_EMPTY_END_OF_CLUSTER_TRIGGER_ARRAY)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CLUSTER_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_LCU_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_UNEXPECTED_INTERRUPT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TIMEOUT_WHILE_WAITING_FOR_INTERRUPT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_SEQUENCER_CONFIGURATION)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_ENABLE_LCU_CONFIGURATION)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_DISABLE_LCU_CONFIGURATION)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_EDGE_CONNECTION_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_EDGE_LAYER_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_BYTES_PER_BUFFER_VALUE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CREDITS_SIZE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_EDGE_LAYER_ALREADY_SET)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_EXPECTED_INTERRUPT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_PCIE_CONTROL_INTERRUPT_ENABLED_DURING_RUN_TIME_MODE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CONFIG_MODULE_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_WAIT_FOR_INTERRUPT_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_EDGE_LAYER_MEMORY_OUT_OF_BOUNDS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_EDGE_LAYER_COMMUNICATION_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_KERNEL_DONE_ADDRESS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_BUFFERS_PER_FRAME_VALUE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_MULTIPLE_EDGE_LAYERS_FOR_SAME_STREAM)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_MULTIPLE_STREAMS_USE_THE_SAME_CHANNEL)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CONTEXT_CREDITS_BY_BYTES)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_REQUESTED_DESCRIPTORS_EXCEEDED_MAX_ALLOWED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_MISMATCH_BETWEEN_EXPECTED_DESCRIPTOR_COUNT_AND_ACTUAL_REQUESTS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_APPLICATION_SET_BEFORE_INIT_STATE_DONE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_APPLICATION_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_HOST_DESCRITPOR_BASE_ADDRESS_IS_NOT_64KB_ALIGNED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_EDGE_LAYER_DIRECTION)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_EDGE_LAYER_CREDIT_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_ADDING_CREDITS_IS_ALLOWED_ONLY_FOR_EDGE_LAYER_DIRECTION_HOST_TO_DEVICE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_PCIE_CHANNEL_INDEX_AND_DIRECTION_MISMATCH)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_ACTION_LIST_OFFSET)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_CHANGING_APP_IS_NOT_ALLOWED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_BATCH_SIZE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_CONFIG_BREAKPOINT_BEFORE_INIT_STATE_DONE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_APPLICATION_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_BATCH_INDEX_FOR_SINGLE_CONTEXT_APPS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_CONTEXT_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_BREAKPOINT_ID)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_BREAKPOINT_CONTROL)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_RECEIVED_INVALID_BATCH_INDEX_AND_CONTEXT_INDEX_COMBO)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_USER_MUST_CLEAR_OR_REACH_BREAKPOINT_BEFORE_SETTING_NEW_ONE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_USER_CANT_CONTINUE_WHEN_BREAKPOINT_IS_CLEARED_OR_NOT_REACHED_YET)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_CANT_FIND_CONTEXT_DUMMY_STREAM)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TRYING_TO_READ_MAIN_HEADER_BEFORE_INIT_STATUS_DONE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_WAIT_FOR_INPUT_CREDITS_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_COULD_NOT_FIND_MATCH_APP_FOR_CONTEXT_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_EDGE_LAYERS_ON_MULTIPLE_CONTROLS_IS_NOT_SUPPORTED_YET)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_STREAM_CREDIT_NOT_ZERO_AT_END_OF_CONTEXT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_FIFO_COUNTER_NOT_ZERO_AT_END_OF_CONTEXT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_EXECUTE_ACTION_WAS_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_FAILED_TO_RESET_STATE_MACHINE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_SEQUENCER_ABBALE_VALIDATION_FAILED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_USER_TIMESTAMP_CONFIGURATION_IS_DISABLED_ALREADY)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_DESC_PAGE_SIZE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_VDMA_CHANNEL_DEPTH)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_CANT_CONFIGURE_HEF_WHILE_ACTIVATED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_DDR_BUFFER_TASK_IS_NOT_IDLE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_DATAFLOW_IS_ACTIVE_ON_ANOTHER_MANAGER)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TEMPERATURE_SAFE_STATE_UNINITALIZED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_HIGH_TEMPERATURE_WHEN_OPENING_STREAM)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_NETWORK_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_REPEATED_ACTION_INVALID_HEADER)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_REPEATED_ACTION_INVALID_SUBACTION_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_REPEATED_ACTION_INVALID_ACTION_COUNT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_CURRENT_SAFE_STATE_UNINITALIZED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_HIGH_CURRENT_WHEN_OPENING_STREAM)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_WAIT_FOR_PREDICATE_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_WAIT_FOR_DMA_IDLE_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_ACTION_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_ACTION_IS_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CFG_CHANNELS_COUNT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_HOST_BUFFER_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_BURST_CREDITS_TASK_IS_NOT_IDLE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_KERNEL_COUNT_OVERFLOW)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CONFIG_STREAM_INDEX)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_ADD_ACTION_TO_BATCH_SWITCH_BUFFER_REACHED_FORBIDDEN_MEMORY_SPACE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_TASK_STATUS_WAIT_FOR_INTERRUPT_INTERRUPTED_BY_BATCH_CHANGE_REQUEST)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_CANT_CLEAR_CONFIGURED_APPS_WHILE_ACTIVATED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CONTEXT_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_UNEXPECTED_CONTEXT_ORDER)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_DYNAMIC_CONTEXT_COUNT)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_CONTEXT_INDEX_OUT_OF_RANGE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TOTAL_PROVIDED_EDGE_LAYERS_LARGER_THEN_EXPECTED)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_REACHED_TIMEOUT_WHILE_WAITING_FOR_NETWORK_IDLE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_WRITE_DATA_BY_TYPE_ACTION_INVALID_TYPE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_WRITE_DATA_BY_TYPE_ACTION_INVALID_MEMORY_SPACE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_REACHED_TIMEOUT_WHILE_WAITING_FOR_BATCH_SWITCH_CONTEXT_TO_END)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_EXTERNAL_ACTION_LIST_ADDRESS)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_CACHE_SIZE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_READ_OFFSET_SIZE)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_INVALID_SLEEP_TIME)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_SRAM_MEMORY_FULL)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TIMEOUT_LCU)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TIMEOUT_SEQUENCER)\
   FIRMWARE_STATUS__X(CONTEXT_SWITCH_STATUS_TIMEOUT_CONFIG_DONE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__D2H_EVENT_MANAGER)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_MESSAGE_HIGH_PRIORITY_QUEUE_CREATE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_MESSAGE_QUEUE_CREATE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_MESSAGE_LOW_PRIORITY_QUEUE_CREATE_FAILED)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_INVALID_MESSAGE_QUEUE_HANDLE)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_SENDING_MESSAGE_FAIL)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_SEND_EVENT_OVER_UDP_FAIL)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_SEND_EVENT_OVER_VDMA_FAIL)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_INVALID_COMMUNICATION_TYPE)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_VDMA_COMMUNICATION_NOT_ACTIVE)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_INVALID_PRIORITY_QUEUE_HANDLE)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_INIT_UDP_FAIL)\
   FIRMWARE_STATUS__X(HAILO_D2H_EVENT_MANAGER_STATUS_INVALID_PARAMETERS)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__WD)\
   FIRMWARE_STATUS__X(WD_STATUS_INVALID_PARAMETERS)\
   FIRMWARE_STATUS__X(WD_STATUS_CONFIG_WHILE_ENABLED)\
   FIRMWARE_STATUS__X(WD_STATUS_ENABLE_WHILE_ENABLED) /* DEPRECATED */\
   FIRMWARE_STATUS__X(WD_STATUS_DISABLE_WHILE_DISABLED) /* DEPRECATED */\
   FIRMWARE_STATUS__X(WD_STATUS_INVALID_WD_CYCLES)\
   FIRMWARE_STATUS__X(WD_SERVICE_INIT_FAILED)\
   FIRMWARE_STATUS__X(WD_SERVICE_WD_UNSUPPORTED)\
   FIRMWARE_STATUS__X(WD_SERVICE_WD_CONFIG_AND_ENABLE_CPU_ID_INVALID)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__HEALTH_MONITOR)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_CLOSING_STREAMS_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_SENDING_CLOSED_STREAM_EVENT_TO_HOST_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_SENDING_ALARM_A_EVENT_TO_HOST_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_IS_NOT_INITIALIZED_SUCCESSFULLY)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_CANT_OPEN_STREAM_TS0_TOO_HOT) /* DEPRECATED - See TEMPERATURE_PROTECTION */\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_CANT_OPEN_STREAM_TS1_TOO_HOT) /* DEPRECATED - See TEMPERATURE_PROTECTION */\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_UNKNOWN_ALARM_TYPE)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_READING_TEMPERATURE_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_FAILED_TO_CREATE_QUEUE)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_OVERCURRENT_ALERT_ACTION_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_SENDING_OVERCURRENT_ALARM_EVENT_TO_HOST_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_QUEUEING_OVERCURRENT_MESSAGE_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_QUEUEING_CORE_RESET_MESSAGE_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_QUEUEING_SOFT_RESET_MESSAGE_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_INVALID_OVERCURRENT_OVERCURRENT_ZONE) /* DEPRECATED -*/\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_INVALID_MESSAGE_TYPE)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_INVALID_TEMPERATURE_ALARM_TYPE) /* DEPRECATED - See TEMPERATURE_PROTECTION */\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_SETUP_SAFETY_INTERRUPTS_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_QUEUEING_RESET_CONTEXT_SWITCH_MESSAGE_FAILED)\
   FIRMWARE_STATUS__X(HEALTH_MONITOR_QUEUEING_TEMPERATURE_MESSAGE_FAILED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__PVT_DRIVER)\
   FIRMWARE_STATUS__X(PVT_DRIVER_WRONG_PVTC_COMPONENT_ID)\
   FIRMWARE_STATUS__X(PVT_DRIVER_WRONG_PVTC_IP_NUMBER)\
   FIRMWARE_STATUS__X(PVT_DRIVER_INVALID_SCRATCH_RESET_VALUE)\
   FIRMWARE_STATUS__X(PVT_DRIVER_INVALID_SCRATCH_TEST_VALUE)\
   FIRMWARE_STATUS__X(PVT_DRIVER_TS_SDIF_IS_BUSY)\
   FIRMWARE_STATUS__X(PVT_DRIVER_TS_SDIF_IS_LOCKED)\
   FIRMWARE_STATUS__X(PVT_DRIVER_ISR_TS_FAULT)\
   FIRMWARE_STATUS__X(PVT_DRIVER_IS_NOT_INITIALIZED_SUCCESSFULLY)\
   FIRMWARE_STATUS__X(PVT_INVALID_CLOCK_FREQUENCY)\
   FIRMWARE_STATUS__X(PVT_SDIF_DATA_FAULT)\
   FIRMWARE_STATUS__X(PVT_WRITE_PD_SDIF_FAIL_TO_WRITE_IP_CFG)\
   FIRMWARE_STATUS__X(PVT_WRITE_PD_SDIF_FAIL_TO_WRITE_IP_CTRL)\
   FIRMWARE_STATUS__X(PVT_WRITE_PD_SMPL_CNT_FAILED_TO_INCREMENT)\
   FIRMWARE_STATUS__X(PVT_PD_NOT_INITIALIZED_SUCCESSFULLY)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__ISP)\
   FIRMWARE_STATUS__X(ISP_UNSUPPORTED_OUTPUT_DATA_TYPE)\
   FIRMWARE_STATUS__X(ISP_UNSUPPORTED_INPUT_DATA_TYPE)\
   FIRMWARE_STATUS__X(ISP_UNSUPPORTED_INPUT_BAYER_ORDER)\
   FIRMWARE_STATUS__X(ISP_WRITE_OUT_OF_PAGE_BOUNDS)\
   FIRMWARE_STATUS__X(ISP_UNSUPPORTED_OPERATION_TYPE)\
   FIRMWARE_STATUS__X(ISP_WRITE_VALUE_TOO_BIG)\
   FIRMWARE_STATUS__X(ISP_AUTO_FOCUS_UNSUPPORTED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__BOARD_CONFIG)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_MAGIC)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_COMMON_HEADER)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_INTERNAL_USE_AREA)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_BOARD_INFO_AREA)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_BOARD_CONFIG)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_SECTION_ERASE_FAILED)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_READ_FROM_EEPROM_FAILED)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_READ_FROM_FLASH_FAILED)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_MAX_NEURAL_NETWORK_CORE_CLOCK_RATE) /* DEPRECATED */\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_NULL_POINTER_PASSED)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_INVALID_STRUCT_SIZE_PASSED)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_READ_FROM_INVALID_STORAGE)\
   FIRMWARE_STATUS__X(BOARD_CONFIG_STATUS_READ_FROM_PCIE_PRELOAD_FAILED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__FIRMWARE_CONFIGS)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_CONFIG_WRITE_NOT_ALLOWED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_USER_CONFIG_WRITE_NOT_ALLOWED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_CONFIG_NOT_LOADED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_USER_CONFIG_NOT_LOADED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_CONFIG_OVERRUN)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_USER_CONFIG_OVERRUN)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_USER_CONFIG_LOAD_NOT_ALLOWED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_CONFIG_READ_PCIE_PRELOAD_FAILED_TO_GET_BOOT_TYPE)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_CONFIG_READ_PCIE_PRELOAD_FAILED_NOT_BOOT_FROM_PCIE)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_INVALID_BOARD_CONFIG_MAGIC_WAS_RECEIVED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_INVALID_BOARD_CONFIG_COMMON_HEADER_WAS_RECEIVED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_USER_CONFIG_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_CONFIG_NULL_ARGUMENT_PASSED)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_BOARD_PCIE_PRELOAD_NOT_ACCESSIBLE_AFTER_BOOT)\
   FIRMWARE_STATUS__X(FIRMWARE_CONFIGS_STATUS_USER_PCIE_PRELOAD_NOT_ACCESSIBLE_AFTER_BOOT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__GPIO)\
   FIRMWARE_STATUS__X(GPIO_BAD_GPIO_INDEX)\
   FIRMWARE_STATUS__X(GPIO_BAD_PINMUX_GROUP)\
   FIRMWARE_STATUS__X(GPIO_SETUP_PINMUX_NOT_SUPPORTED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__OVERCURRENT_PROTECTION)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_INVALID_ALERT_THRESHOLD_VALUE) /* DEPRECATED */\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_INVALID_SAMPLING_PERIOD_VALUE)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_INVALID_AVERAGING_FACTOR_VALUE)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_UNSUPPORTED_SENSOR_TYPE)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_MODULE_IS_NOT_INITIALIZED)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_ALREADY_ACTIVE)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_IS_NOT_ACTIVE)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_INVALID_BOARD_CONFIG_VALUES)\
   FIRMWARE_STATUS__X(OVERCURRENT_PROTECTION_NULL_POINTER_PASSED) /* DEPRECATED */\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__POWER)\
   FIRMWARE_STATUS__X(POWER_INVALID_CONVERSION_TYPE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__SECURE_DEBUG)\
   FIRMWARE_STATUS__X(SOC_ID_COMPUTATION_ERROR)\
   FIRMWARE_STATUS__X(LCS_GET_FAILED)\
   FIRMWARE_STATUS__X(SECURE_SERVICE_MODULE_IS_NOT_INITIALIZED)\
   FIRMWARE_STATUS__X(GOT_INVALID_LCS_FROM_CC)\
   FIRMWARE_STATUS__X(WRONG_LCS_FOR_RMA_ENTRANCE)\
   FIRMWARE_STATUS__X(DEVELOPER_CERTIFICATE_VERIFICATION_FAILED)\
   FIRMWARE_STATUS__X(DEVELOPER_CERTIFICATE_INCOMPATIBLE_FOR_RMA_ENTRANCE)\
   FIRMWARE_STATUS__X(RMA_ENTRANCE_FAILED)\
   FIRMWARE_STATUS__X(DEBUG_CERTIFICATE_TYPE_MISMATCH)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__POWER_STATE)\
   FIRMWARE_STATUS__X(POWER_STATE_STATUS_INVALID_STATE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__SOC)\
   FIRMWARE_STATUS__X(SOC_STATUS_NULL_POINTER_PASSED)\
   FIRMWARE_STATUS__X(SOC_STATUS_INVALID_BOOT_SOURCE)\
   FIRMWARE_STATUS__X(SOC_STATUS_SOFT_RESET_FAILED_INITIALIZING_CRYPTOCELL)\
   FIRMWARE_STATUS__X(SOC_INVALID_CLOCK_RATE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__SECOND_STAGE_UPDATE)\
   FIRMWARE_STATUS__X(SECOND_STAGE_UPDATE_STATUS_INVALID_PARAMETERS)\
   FIRMWARE_STATUS__X(SECOND_STAGE_UPDATE_STATUS_BINARY_SIZE_EXCEEDED)\
   FIRMWARE_STATUS__X(SECOND_STAGE_UPDATE_STATUS_MISMATCHING_MD5)\
   FIRMWARE_STATUS__X(SECOND_STAGE_UPDATE_STATUS_CORE_IS_NOT_IDLE)\
   FIRMWARE_STATUS__X(SECOND_STAGE_UPDATE_STATUS_INVALID_SECOND_STAGE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__DATAFLOW_COMMON)\
   FIRMWARE_STATUS__X(DATAFLOW_COMMON_STATUS_INVALID_BYTES_PER_BUFFER_VALUE)\
   FIRMWARE_STATUS__X(DATAFLOW_COMMON_STATUS_INVALID_BUFFERS_PER_FRAME_VALUE)\
   FIRMWARE_STATUS__X(DATAFLOW_COMMON_STATUS_INVALID_EDGE_LAYER_INDEX)\
   FIRMWARE_STATUS__X(DATAFLOW_COMMON_STATUS_INVALID_PARAMETER)\
   FIRMWARE_STATUS__X(DATAFLOW_COMMON_STATUS_PADDING_NOT_SUPPORTED_FOR_ARCH)\
   FIRMWARE_STATUS__X(DATAFLOW_COMMON_STATUS_INVALID_MAX_BUFFER_PADDING_VALUE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__RESET_HANDLER)\
   FIRMWARE_STATUS__X(RESET_HANDLER_CHIP_RESET_FAILED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__TEMPERATURE_PROTECTION)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_INVALID_TEMPERATURE_ZONE)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_NULL_POINTER_PASSED)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_TS0_TOO_HOT)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_TS1_TOO_HOT)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_READING_TEMPERATURE_FAILED)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_THROTTLING_ACTIVATION_IS_ILLEGAL)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_TEMPERATURE_ZONE_DECREASED)\
   FIRMWARE_STATUS__X(TEMPERATURE_PROTECTION_TEMPERATURE_ZONE_DIDNT_CHANGE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__DDR_BUFFER)\
   FIRMWARE_STATUS__X(DDR_BUFFER_STATUS_TRYING_TO_ADD_DDR_PAIR_WHILE_NOT_IN_IDLE_STATE)\
   FIRMWARE_STATUS__X(DDR_BUFFER_STATUS_TRYING_TO_CHANGE_STATE_TO_INFER_WHILE_ALREADY_DURING_INFER)\
   FIRMWARE_STATUS__X(DDR_BUFFER_STATUS_NO_DDR_PAIRS_TO_RUN_INFER_ON)\
   FIRMWARE_STATUS__X(DDR_BUFFER_STATUS_INFER_REACHED_TIMEOUT)\
   FIRMWARE_STATUS__X(DDR_BUFFER_STATUS_OVERFLOW_IN_DESCRIPTORS_PER_BATCH)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__PL320)\
   FIRMWARE_STATUS__X(PL320_MAILBOX_BUSY)\
   FIRMWARE_STATUS__X(PL320_MAILBOX_MAXIMUM_CHANNELS)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__PROCESS_MONITOR)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_ACTIVATE_FAILED)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_NULL_POINTER_PASSED)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_INVALID_BUFFER_SIZE)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_PVT_PD_UNINITIALIZED)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_PVT_GET_PD_RESULT_FAILED_FOR_DELAY_CHAIN)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_PVT_POWER_DOWN_FAILED)\
   FIRMWARE_STATUS__X(PROCESS_MONITOR_INVALID_DELAY_CHAIN_INDEX)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__MBIST)\
   FIRMWARE_STATUS__X(MBIST_STATUS_INVALID_ARGS)\
   FIRMWARE_STATUS__X(MBIST_STATUS_TRYING_TO_READ_BISR_REQUIRED_BEFORE_MBIST_INIT_DONE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__NN_CONFIG_SERVICE)\
   FIRMWARE_STATUS__X(NN_CONFIG_SERVICE_STATUS_INVALID_CONFIG_STREAM_INDEX)\
   FIRMWARE_STATUS__X(NN_CONFIG_SERVICE_STATUS_CSM_FETCH_ACTIONS_REQUEIRE_FW_TO_BE_COMPILED_WITH_CSM_SUPPORT)\
   FIRMWARE_STATUS__X(NN_CONFIG_SERVICE_STATUS_CSM_FETCH_ACTIONS_REQUEIRE_CSM_IN_PRE_FETCH_MODE)\
   FIRMWARE_STATUS__X(NN_CONFIG_SERVICE_STATUS_CSM_NOT_ENABLED_WHILE_TRYING_TO_FETCH_CONFIG)\
   FIRMWARE_STATUS__X(NN_CONFIG_SERVICE_STATUS_CSM_BURST_COUNTER_IS_NOT_ZERO)\
   FIRMWARE_STATUS__X(NN_CONFIG_SERVICE_STATUS_CSM_CREDIT_COUNTER_IS_NOT_ZERO)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__CONFIG_MANAGER_WRAPPER)\
   FIRMWARE_STATUS__X(CONFIG_MANAGER_WRAPPER_STATUS_ACTION_TYPE_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(CONFIG_MANAGER_WRAPPER_STATUS_NULL_ARG_PASSED)\
   FIRMWARE_STATUS__X(CONFIG_MANAGER_WRAPPER_STATUS_WAIT_FOR_INTERRUPT_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(CONFIG_MANAGER_WRAPPER_STATUS_RECEIVED_UNEXPECTED_INTERRUPT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__CSM_CONFIG_MANAGER)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_NOT_IMPLEMENTED)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_INVALID_CONFIG_STREAM_INDEX)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_INVALID_CONFIG_MODULE_INDEX)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_NULL_ARG_PASSED)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_CSM_FETCH_ACTIONS_REQUEIRE_CSM_IN_PRE_FETCH_MODE)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_CSM_NOT_ENABLED_WHILE_TRYING_TO_FETCH_CONFIG)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_CSM_BURST_COUNTER_IS_NOT_ZERO)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_CSM_CREDIT_COUNTER_IS_NOT_ZERO)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_CSM_FIFO_NOT_EMPTY)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_INVALID_HOST_PAGE_SIZE)\
   FIRMWARE_STATUS__X(CSM_CONFIG_MANAGER_STATUS_INVALID_BUFFER_SIZE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__PCIE_CONFIG_MANAGER)\
   FIRMWARE_STATUS__X(PCIE_CONFIG_MANAGER_STATUS_NOT_IMPLEMENTED)\
   FIRMWARE_STATUS__X(PCIE_CONFIG_MANAGER_STATUS_INVALID_PCIE_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(PCIE_CONFIG_MANAGER_STATUS_INVALID_MODULE_INDEX)\
   FIRMWARE_STATUS__X(PCIE_CONFIG_MANAGER_STATUS_NULL_ARG_PASSED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__VDMA_SERVICE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_H2D_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_D2H_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_HOST_DESCRITPOR_BASE_ADDRESS_IS_NOT_64KB_ALIGNED)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_VDMA_CHANNEL_DEPTH)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_CHANNEL_CANNOT_FETCH_DESCRITPROTS_WHILE_ABORTED)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_DESCRIPTOR_COUNT_INDICATE_NOT_ALL_DATA_WAS_TRANSFERRED)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_CHANNEL_TYPE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_VDMA_MUST_BE_DISABLED_WHEN_CLEARING_CHANNEL_CREDITS)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_CHANNEL_NOT_IDLE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_NOT_IMPLEMENTED)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_H2D_CREDITS)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_D2H_CREDITS)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_NULL_ARG_PASSED)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_CHANNEL_FAILED_TO_REACH_IDLE_STATE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_VDMA_MUST_BE_STOPPED_WHEN_CHECKING_IDLE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_EXTERNAL_DESC_COUNT_MUST_BE_POWER_OF_2)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_TOO_MANY_DESCRIPTORS)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_HOST_BUFFER_TYPE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_DESC_PAGE_SIZE_MUST_BE_POWER_OF_2)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INITIAL_DESC_BIGGER_EQ_THAN_TOTAL)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_CCB_NOT_IMPLEMENTED_OVER_PCIE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_CCB_BASE_ADDRESS_IS_NOT_IN_MASK)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INITIAL_DESC_BIGGER_THAN_TOTAL)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_INITIAL_CREDIT_SIZE)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_TOO_LARGE_BYTES_IN_PATTERN)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_ENGINE_INDEX)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_CONSTANTS)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_STATUS_INVALID_EDGE_LAYER_DIRECTION)\
   FIRMWARE_STATUS__X(VDMA_SERVICE_INSUFFICIENT_DESCRIPTORS_COUNT)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__MEMORY_LOGGER)\
   FIRMWARE_STATUS__X(MEMORY_LOGGER_STATUS_DEBUG_INSUFFICIENT_MEMORY)\
   FIRMWARE_STATUS__X(MEMORY_LOGGER_STATUS_GOT_UNALIGNED_ADDRESS_FOR_PCIE_BUFFERS)\
   FIRMWARE_STATUS__X(MEMORY_LOGGER_STATUS_INVALID_DEBUG_CHIP_OFFSET)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__UART_LOGGER)\
   FIRMWARE_STATUS__X(UART_LOGGER_STATUS_QUEUE_SEND_FAILED)\
   FIRMWARE_STATUS__X(UART_LOGGER_STATUS_FAILED_ACQUIRE_SEMAPHORE)\
   FIRMWARE_STATUS__X(UART_LOGGER_STATUS_FAILED_RELEASE_SEMAPHORE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__DRAM_DMA_SERVICE)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_PARAMETERS)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_SETUP_INTERRUPT_HANDLER_FAILED)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_BURST_CREDIT_SIZE_TOO_BIG)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_CHANNEL_DMA_ADDRESS)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_DESC_PAGE_SIZE)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_NUM_PAGES_IS_OUT_OF_RANGE)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_INVALID_INITIAL_CREDIT_SIZE)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_TOTAL_DESCS_COUNT_IS_OUT_OF_RANGE)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_TOTAL_DESCS_COUNT_MUST_BE_POWER_OF_2)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_INVALID_DESCS_COUNT)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_DESC_PER_INTERRUPT_NOT_IN_MASK)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_ENGINE_INDEX)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_QM_INDEX)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_CHANNEL_TYPE)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_PERIPH_BYTES_PER_BUFFER)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_BYTES_IN_PATTERN)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_STREAM_INDEX)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_INVALID_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_FAILED_TO_RESET_QM_CREDITS)\
   FIRMWARE_STATUS__X(DRAM_DMA_SERVICE_STATUS_TRIED_USING_BURST_IN_NOT_ENHANCED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__NN_CORE_SERVICE)\
   FIRMWARE_STATUS__X(NN_CORE_SERVICE_STATUS_INVALID_ARG_PASSED)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__DATA_STREAM_MANAGER_WRAPPER)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_NOT_IMPLEMENTED)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_INVALID_EDGE_LAYER_INDEX)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_INVALID_DESC_PAGE_SIZE)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_INVALID_EDGE_LAYER_DIRECTION)\
   FIRMWARE_STATUS__X(DATA_STREAM_WRAPPER_STATUS_INVALID_CHANNEL_INDEX)\
   FIRMWARE_STATUS__X(DATA_STREAM_WRAPPER_STATUS_INVALID_STREAM_INDEX)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_STATUS_INVALID_CREDIT_TYPE)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_INVALID_HOST_BUFFER_TYPE)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_STATUS_BATCH_CREDITS_OVERFLOW)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_BURST_CREDIT_TASK_MUST_BE_DISABLED_WHILE_CHANGING_BATCH)\
   FIRMWARE_STATUS__X(DATA_STREAM_MANAGER_WRAPPER_STATUS_UNABLE_TO_RESET_FRAME_COUNTER)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__BURST_CREDITS_TASK)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TRYING_TO_ADD_ACTION_WHILE_NOT_IN_IDLE_STATE)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TOO_MANY_ACTIONS)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TRYING_TO_CHANGE_STATE_TO_INFER_WHILE_ALREADY_IN_INFER)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_INFER_REACHED_TIMEOUT)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TASK_DEACTIVATED)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_FAILED_TO_FIND_STREAM_INDEX)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TASK_NO_CONFIGURED_ACTIONS)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TASK_EXPECTED_HIGHER_BATCH)\
   FIRMWARE_STATUS__X(BURST_CREDITS_TASK_STATUS_TASK_REACHED_TIMEOUT_WAITING_FOR_DEACTIVATION)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__TASK_SYNC_EVENTS)\
   FIRMWARE_STATUS__X(TASK_SYNC_EVENTS_STATUS_START_TASK_WHILE_IT_IS_RUNNING)\
   FIRMWARE_STATUS__X(TASK_SYNC_EVENTS_STATUS_START_TASK_WHILE_TASK_NOT_DONE)\
   FIRMWARE_STATUS__X(TASK_SYNC_EVENTS_STATUS_FAILED_TO_RESET_STATE_MACHINE)\
   FIRMWARE_STATUS__X(TASK_SYNC_EVENTS_STATUS_DONE_TASK_WHILE_IT_IS_NOT_RUNNING)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__NMS_MANAGER)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_ARGUMENT)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_WAIT_FOR_INTERRUPT_INTERRUPTED_BY_RESET_REQUEST)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_AGGREGATOR_INDEX)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_OUTPUT_BUFFER_INDEX)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_OUTPUT_BUFFER_CLUSTER_INDEX)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_OUTPUT_BUFFER_INTERFACE)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_NOT_SUPPORTED_OPERATION)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_NETWORK_INDEX)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_NMS_UNIT_INDEX)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_BATCH_SIZE)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_NUM_CLASSES_SIZE)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_BURST_SIZE)\
   FIRMWARE_STATUS__X(NMS_MANAGER_STATUS_INVALID_LAST_FRAME_IN_BATCH_SIZE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__CLUSTER_MANAGER)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_CLUSTER_INDEX)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_INITIAL_L3_CUT)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_INITIAL_L3_OFFSET)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_LCU_INDEX)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_KERNEL_DONE_ADDRESS)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_RECEIVED_UNEXPECTED_INTERRUPT)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_NETWORK_INDEX)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_KERNEL_DONE_COUNT)\
   FIRMWARE_STATUS__X(CLUSTER_MANAGER_STATUS_INVALID_EXTENSION)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__HW_INFER_MANAGER)\
   FIRMWARE_STATUS__X(HW_INFER_MANAGER_STATUS_NETWORK_GROUP_NOT_CONFIGURED_BEFORE_INFER_START)\
   FIRMWARE_STATUS__X(HW_INFER_MANAGER_STATUS_NETWORK_GROUP_ALREADY_ACTIVATED)\
   FIRMWARE_STATUS__X(HW_INFER_MANAGER_STATUS_STATE_MACHINE_NOT_IN_RESET_STATE_BEFORE_DEACTIVATE)\
   FIRMWARE_STATUS__X(HW_INFER_MANAGER_STATUS_INVALID_STATE)\
   FIRMWARE_STATUS__X(HW_INFER_MANAGER_STATUS_INVALID_BOUNDARY_CHANNEL_MODE)\
   \
   FIRMWARE_MODULE__X(FIRMWARE_MODULE__INFINITE_CONTEXT_LOADER)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_EVENT_BITS_NOT_CLEARED_BEFORE_COPY_CALL)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_TIMEOUT_OCCURED_WAITING_FOR_COPY)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_NOT_SUPPORTED)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_NOT_MODULE_NOT_INITIALIZED)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_QUEUE_SEND_FAIL)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_QUEUE_RECEIVE_FAIL)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_QUEUE_FULL)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_QUEUE_FAILED_INIT)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_NULL_POINTER)\
   FIRMWARE_STATUS__X(INFINITE_CONTEXT_LOADER_STATUS_INVALID_NUM_CONTEXTS)\

typedef enum {
#define FIRMWARE_MODULE__X(module) module,
#define FIRMWARE_STATUS__X(name)
    FIRMWARE_STATUS__VARIABLES
#undef FIRMWARE_STATUS__X
#undef FIRMWARE_MODULE__X

    /* Must be last! */
    FIRMWARE_MODULE_COUNT
} FIRMWARE_STATUS__modules_t;

/* Validate each field at the status is in range range */
CASSERT(FIRMWARE_STATUS__COMPONENT_ID<=FIRMWARE_STATUS__COMPONENT_MAX, firmware_status_h);
CASSERT(FIRMWARE_MODULE_COUNT<FIRMWARE_STATUS__MODULE_INDEX_MAX, firmware_status_h);

typedef enum {
    HAILO_STATUS_SUCCESS = 0,

#define FIRMWARE_MODULE__X(module) module##_START = FIRMWARE_STATUS__COMPONENT_SHIFT(FIRMWARE_STATUS__COMPONENT_ID) \
    | FIRMWARE_STATUS__MODULE_INDEX_SHIFT(module),
#define FIRMWARE_STATUS__X(name) name, 
    FIRMWARE_STATUS__VARIABLES
#undef FIRMWARE_STATUS__X
#undef FIRMWARE_MODULE__X
} FIRMWARE_STATUS_t;

HAILO_COMMON_STATUS_t FIRMWARE_STATUS__get_textual(FIRMWARE_STATUS_t fw_status, const char **text);

#ifdef __cplusplus
}
#endif

#endif /* __FIRMWARE_STATUS_H__ */
