//! # block-device-id
//!
//! Retrieve innate unique identifiers from block devices.
//!
//! This crate provides functionality to identify block devices by their
//! hardware-innate identifiers, such as MMC CID or NVMe controller serial
//! numbers.
//!
//! ## Example
//!
//! ```no_run
//! use block_device_id::get_id_by_path;
//!
//! let id = get_id_by_path("/dev/nvme0n1").unwrap();
//! println!("{}", id.id.to_string_lossy());
//! ```

mod device;
mod sysfs;

pub mod ffi;

use std::ffi::OsString;
use std::io;
use std::path::Path;

pub use device::BlockDeviceType;

/// Result of successfully identifying a block device.
#[derive(Debug, Clone, PartialEq, Eq)]
pub struct BlockDeviceId {
    /// The type of block device.
    pub device_type: BlockDeviceType,
    /// The unique hardware-innate identifier (raw bytes from sysfs).
    pub id: OsString,
}

/// Errors that can occur when retrieving a block device ID.
#[derive(Debug)]
pub enum Error {
    /// The path does not refer to a block device.
    NotABlockDevice,
    /// The device type is not supported (only MMC and NVMe are supported).
    UnsupportedDevice,
    /// The device type was recognised but no innate identifier is available.
    NoIdentifierAvailable,
    /// The identifier was too large and would have been truncated.
    IdentifierTruncated,
    /// An I/O error occurred.
    Io(io::Error),
}

impl std::fmt::Display for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Error::NotABlockDevice => write!(f, "not a block device"),
            Error::UnsupportedDevice => {
                write!(f, "unsupported device type (only MMC and NVMe are supported)")
            }
            Error::NoIdentifierAvailable => {
                write!(f, "device type recognised but no innate identifier available")
            }
            Error::IdentifierTruncated => {
                write!(f, "device identifier too large (would be truncated)")
            }
            Error::Io(e) => write!(f, "I/O error: {}", e),
        }
    }
}

impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        match self {
            Error::Io(e) => Some(e),
            _ => None,
        }
    }
}

impl From<io::Error> for Error {
    fn from(e: io::Error) -> Self {
        Error::Io(e)
    }
}

/// Get the innate unique identifier for a block device by path.
///
/// Symlinks (e.g., `/dev/disk/by-label/rootfs`) are followed automatically.
///
/// # Arguments
///
/// * `path` - Path to a block device or symlink to one.
///
/// # Returns
///
/// Returns the device type and unique identifier on success.
///
/// # Errors
///
/// * `Error::NotABlockDevice` - The path does not refer to a block device.
/// * `Error::UnsupportedDevice` - Device type not supported (only MMC/NVMe).
/// * `Error::NoIdentifierAvailable` - Device recognised but no ID found.
/// * `Error::IdentifierTruncated` - ID too large for internal buffer.
/// * `Error::Io` - System call or file operation failed.
///
/// # Example
///
/// ```no_run
/// use block_device_id::get_id_by_path;
///
/// let id = get_id_by_path("/dev/nvme0n1p1").unwrap();
/// println!("{}", id.id.to_string_lossy());
/// ```
pub fn get_id_by_path<P: AsRef<Path>>(path: P) -> Result<BlockDeviceId, Error> {
    device::get_block_device_id_by_path(path.as_ref())
}
