"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatRoutes = registerChatRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _olly_chat_service = require("../services/chat/olly_chat_service");
var _agent_framework_storage_service = require("../services/storage/agent_framework_storage_service");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
var _error_handler = require("./error_handler");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const llmRequestRoute = {
  path: _llm.ASSISTANT_API.SEND_MESSAGE,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.maybe(_configSchema.schema.string()),
      messages: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
      input: _configSchema.schema.object({
        type: _configSchema.schema.literal('input'),
        context: _configSchema.schema.object({
          appId: _configSchema.schema.maybe(_configSchema.schema.string()),
          content: _configSchema.schema.maybe(_configSchema.schema.string()),
          datasourceId: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        content: _configSchema.schema.string(),
        contentType: _configSchema.schema.literal('text'),
        promptPrefix: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
      nextToken: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const abortAgentExecutionRoute = {
  path: `${_llm.ASSISTANT_API.ABORT_AGENT_EXECUTION}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const regenerateRoute = {
  path: `${_llm.ASSISTANT_API.REGENERATE}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string(),
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationsRoute = {
  path: _llm.ASSISTANT_API.CONVERSATIONS,
  validate: {
    query: _configSchema.schema.object({
      perPage: _configSchema.schema.number({
        min: 0,
        defaultValue: 20
      }),
      page: _configSchema.schema.number({
        min: 0,
        defaultValue: 1
      }),
      sortOrder: _configSchema.schema.maybe(_configSchema.schema.string()),
      sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
      fields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
      search: _configSchema.schema.maybe(_configSchema.schema.string()),
      searchFields: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const deleteConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const updateConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      title: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getTracesRoute = {
  path: `${_llm.ASSISTANT_API.TRACE}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const feedbackRoute = {
  path: `${_llm.ASSISTANT_API.FEEDBACK}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      satisfaction: _configSchema.schema.boolean()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const accountRoute = {
  path: `${_llm.ASSISTANT_API.ACCOUNT}`,
  validate: {}
};
function registerChatRoutes(router, routeOptions) {
  const createStorageService = async (context, dataSourceId) => new _agent_framework_storage_service.AgentFrameworkStorageService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }), routeOptions.messageParsers);
  const createChatService = async (context, dataSourceId) => new _olly_chat_service.OllyChatService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }));
  router.post(llmRequestRoute, async (context, request, response) => {
    var _outputs, _outputs2;
    const {
      messages = [],
      input,
      conversationId: conversationIdInRequestBody
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.requestLLM({
        messages,
        input,
        conversationId: conversationIdInRequestBody
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
    if (outputs.stream) {
      const result = response.ok({
        headers: {
          // Browsers often need to buffer the entire response before decompressing, which defeats the purpose of streaming.
          // need to set 'Content-Encoding' as 'identity' here to prevent browser buffering the response.
          'Content-Encoding': 'identity',
          Connection: 'keep-alive',
          'Content-Type': 'text/event-stream'
        },
        body: outputs.stream
      });
      return result;
    }

    /**
     * Retrieve latest interactions from memory
     */
    const conversationId = ((_outputs = outputs) === null || _outputs === void 0 ? void 0 : _outputs.conversationId) || conversationIdInRequestBody;
    const interactionId = ((_outputs2 = outputs) === null || _outputs2 === void 0 ? void 0 : _outputs2.interactionId) || '';
    try {
      if (!conversationId) {
        throw new Error('Not a valid conversation');
      }
      const resultPayload = {
        messages: [],
        interactions: [],
        conversationId
      };
      if (!conversationIdInRequestBody) {
        /**
         * If no conversationId is provided in request payload,
         * it means it is a brand new conversation,
         * need to fetch all the details including title.
         */
        const conversation = await storageService.getConversation(conversationId);
        resultPayload.interactions = conversation.interactions;
        resultPayload.messages = conversation.messages;
        resultPayload.title = conversation.title;
      } else {
        /**
         * Only response with the latest interaction.
         * It may have some issues in Concurrent case like a user may use two tabs to chat with Chatbot in one conversation.
         * But for now we will ignore this case, can be optimized by always fetching conversation if we need to take this case into consideration.
         */
        const interaction = await storageService.getInteraction(conversationId, interactionId);
        resultPayload.interactions = [interaction].filter(item => item);
        resultPayload.messages = resultPayload.interactions.length ? await storageService.getMessagesFromInteractions(resultPayload.interactions) : [];
      }
      resultPayload.messages.filter(message => message.type === 'input').forEach(msg => {
        // hide additional conetxt to how was it generated
        const index = msg.content.indexOf('answer question:');
        const len = 'answer question:'.length;
        if (index !== -1) {
          msg.content = msg.content.substring(index + len);
        }
      });
      return response.ok({
        body: resultPayload
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationsRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversations(request.query);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      var _error$meta;
      // Return empty result for 404 errors
      if ((error === null || error === void 0 || (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.statusCode) === 404) {
        return response.ok({
          body: {
            objects: [],
            total: 0
          }
        });
      }
      context.assistant_plugin.logger.error(error);
      return (0, _error_handler.handleError)(error, response, context.assistant_plugin.logger);
    }
  });
  router.delete(deleteConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.deleteConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(updateConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.updateConversation(request.params.conversationId, request.body.title);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getTracesRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getTraces(request.params.interactionId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post(abortAgentExecutionRoute, async (context, request, response) => {
    const chatService = await createChatService(context, request.query.dataSourceId);
    try {
      chatService.abortAgentExecution(request.body.conversationId);
      context.assistant_plugin.logger.info(`Abort agent execution: ${request.body.conversationId}`);
      return response.ok();
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(regenerateRoute, async (context, request, response) => {
    const {
      conversationId,
      interactionId
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.regenerate({
        conversationId,
        interactionId
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }

    /**
     * Retrieve latest interactions from memory
     */
    try {
      var _outputs3;
      const interaction = await storageService.getInteraction(conversationId, ((_outputs3 = outputs) === null || _outputs3 === void 0 ? void 0 : _outputs3.interactionId) || '');
      const finalInteractions = [interaction].filter(item => item);
      const messages = finalInteractions.length ? await storageService.getMessagesFromInteractions(finalInteractions) : [];
      return response.ok({
        body: {
          interactions: finalInteractions,
          messages,
          conversationId
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(feedbackRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const {
      interactionId
    } = request.params;
    try {
      const updateResponse = await storageService.updateInteraction(interactionId, {
        feedback: request.body
      });
      return response.ok({
        body: {
          ...updateResponse,
          success: true
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(accountRoute, async (context, request, response) => {
    try {
      var _auth$state$authInfo$, _auth$state;
      const auth = routeOptions.auth.get(request);
      return response.ok({
        body: {
          user_name: (_auth$state$authInfo$ = auth === null || auth === void 0 || (_auth$state = auth.state) === null || _auth$state === void 0 || (_auth$state = _auth$state.authInfo) === null || _auth$state === void 0 ? void 0 : _auth$state.user_name) !== null && _auth$state$authInfo$ !== void 0 ? _auth$state$authInfo$ : _llm.DEFAULT_USER_NAME
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.ok({
        body: {
          user_name: _llm.DEFAULT_USER_NAME
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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