"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssistantPlugin = void 0;
var _operators = require("rxjs/operators");
var _basic_input_output_parser = require("./parsers/basic_input_output_parser");
var _visualization_card_parser = require("./parsers/visualization_card_parser");
var _chat_routes = require("./routes/chat_routes");
var _text2viz_routes = require("./routes/text2viz_routes");
var _assistant_service = require("./services/assistant_service");
var _agent_routes = require("./routes/agent_routes");
var _summary_routes = require("./routes/summary_routes");
var _capabilities_provider = require("./vis_type_nlq/capabilities_provider");
var _saved_object_type = require("./vis_type_nlq/saved_object_type");
var _capabilities = require("./capabilities");
var _constants = require("./utils/constants");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
class AssistantPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "messageParsers", []);
    _defineProperty(this, "assistantService", new _assistant_service.AssistantService());
    this.logger = initializerContext.logger.get();
  }
  async setup(core) {
    this.logger.debug('Assistant: Setup');
    const config = await this.initializerContext.config.create().pipe((0, _operators.first)()).toPromise();
    const assistantServiceSetup = this.assistantService.setup();
    const router = core.http.createRouter();
    core.http.registerRouteHandlerContext('assistant_plugin', () => {
      return {
        config,
        logger: this.logger
      };
    });
    (0, _agent_routes.registerAgentRoutes)(router, assistantServiceSetup);

    // Register server side APIs
    (0, _chat_routes.registerChatRoutes)(router, {
      messageParsers: this.messageParsers,
      auth: core.http.auth
    });

    // Register router for text to visualization
    if (config.text2viz.enabled) {
      (0, _text2viz_routes.registerText2VizRoutes)(router, assistantServiceSetup);
      core.capabilities.registerProvider(_capabilities_provider.capabilitiesProvider);
      core.savedObjects.registerType(_saved_object_type.visNLQSavedObjectType);
    }

    // Register router for alert insight
    if (config.alertInsight.enabled) {
      (0, _summary_routes.registerSummaryAssistantRoutes)(router, assistantServiceSetup);
    }
    core.capabilities.registerProvider(_capabilities.capabilitiesProvider);
    // register UI capabilities from dynamic config service
    core.capabilities.registerSwitcher(async opensearchDashboardsRequest => {
      const dynamicConfigServiceStart = await core.dynamicConfigService.getStartService();
      const store = dynamicConfigServiceStart.getAsyncLocalStore();
      const client = dynamicConfigServiceStart.getClient();
      try {
        const dynamicConfig = await client.getConfig({
          pluginConfigPath: 'assistant'
        }, {
          asyncLocalStorageContext: store
        });
        const [coreStart] = await core.getStartServices();
        const savedObjectsClient = coreStart.savedObjects.getScopedClient(opensearchDashboardsRequest);
        const uiSettingsClient = coreStart.uiSettings.asScopedToClient(savedObjectsClient);
        const isAssistantEnabledBySetting = Boolean(await uiSettingsClient.get(_constants.ENABLE_AI_FEATURES).catch(() => false));
        return {
          assistant: {
            enabled: dynamicConfig.enabled && isAssistantEnabledBySetting,
            chatEnabled: dynamicConfig.chat.enabled && isAssistantEnabledBySetting
          }
        };
      } catch (e) {
        this.logger.error(e);
        return {};
      }
    });

    // Register router for discovery summary
    (0, _summary_routes.registerData2SummaryRoutes)(router, assistantServiceSetup);
    const registerMessageParser = messageParser => {
      const findItem = this.messageParsers.find(item => item.id === messageParser.id);
      if (findItem) {
        throw new Error(`There is already a messageParser whose id is ${messageParser.id}`);
      }
      this.messageParsers.push(messageParser);
    };
    registerMessageParser(_basic_input_output_parser.BasicInputOutputParser);
    registerMessageParser(_visualization_card_parser.VisualizationCardParser);
    return {
      assistantService: assistantServiceSetup,
      registerMessageParser,
      removeMessageParser: parserId => {
        const findIndex = this.messageParsers.findIndex(item => item.id === parserId);
        if (findIndex < 0) {
          this.logger.error(`There is not a messageParser whose id is ${parserId}`);
        }
        this.messageParsers.splice(findIndex, 1);
      }
    };
  }
  start(core) {
    this.logger.debug('Assistant: Started');
    this.assistantService.start();
    return {};
  }
  stop() {
    this.assistantService.stop();
  }
}
exports.AssistantPlugin = AssistantPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfb3BlcmF0b3JzIiwicmVxdWlyZSIsIl9iYXNpY19pbnB1dF9vdXRwdXRfcGFyc2VyIiwiX3Zpc3VhbGl6YXRpb25fY2FyZF9wYXJzZXIiLCJfY2hhdF9yb3V0ZXMiLCJfdGV4dDJ2aXpfcm91dGVzIiwiX2Fzc2lzdGFudF9zZXJ2aWNlIiwiX2FnZW50X3JvdXRlcyIsIl9zdW1tYXJ5X3JvdXRlcyIsIl9jYXBhYmlsaXRpZXNfcHJvdmlkZXIiLCJfc2F2ZWRfb2JqZWN0X3R5cGUiLCJfY2FwYWJpbGl0aWVzIiwiX2NvbnN0YW50cyIsIl9kZWZpbmVQcm9wZXJ0eSIsImUiLCJyIiwidCIsIl90b1Byb3BlcnR5S2V5IiwiT2JqZWN0IiwiZGVmaW5lUHJvcGVydHkiLCJ2YWx1ZSIsImVudW1lcmFibGUiLCJjb25maWd1cmFibGUiLCJ3cml0YWJsZSIsImkiLCJfdG9QcmltaXRpdmUiLCJTeW1ib2wiLCJ0b1ByaW1pdGl2ZSIsImNhbGwiLCJUeXBlRXJyb3IiLCJTdHJpbmciLCJOdW1iZXIiLCJBc3Npc3RhbnRQbHVnaW4iLCJjb25zdHJ1Y3RvciIsImluaXRpYWxpemVyQ29udGV4dCIsIkFzc2lzdGFudFNlcnZpY2UiLCJsb2dnZXIiLCJnZXQiLCJzZXR1cCIsImNvcmUiLCJkZWJ1ZyIsImNvbmZpZyIsImNyZWF0ZSIsInBpcGUiLCJmaXJzdCIsInRvUHJvbWlzZSIsImFzc2lzdGFudFNlcnZpY2VTZXR1cCIsImFzc2lzdGFudFNlcnZpY2UiLCJyb3V0ZXIiLCJodHRwIiwiY3JlYXRlUm91dGVyIiwicmVnaXN0ZXJSb3V0ZUhhbmRsZXJDb250ZXh0IiwicmVnaXN0ZXJBZ2VudFJvdXRlcyIsInJlZ2lzdGVyQ2hhdFJvdXRlcyIsIm1lc3NhZ2VQYXJzZXJzIiwiYXV0aCIsInRleHQydml6IiwiZW5hYmxlZCIsInJlZ2lzdGVyVGV4dDJWaXpSb3V0ZXMiLCJjYXBhYmlsaXRpZXMiLCJyZWdpc3RlclByb3ZpZGVyIiwidmlzTkxRQ2FwYWJpbGl0aWVzUHJvdmlkZXIiLCJzYXZlZE9iamVjdHMiLCJyZWdpc3RlclR5cGUiLCJ2aXNOTFFTYXZlZE9iamVjdFR5cGUiLCJhbGVydEluc2lnaHQiLCJyZWdpc3RlclN1bW1hcnlBc3Npc3RhbnRSb3V0ZXMiLCJjYXBhYmlsaXRpZXNQcm92aWRlciIsInJlZ2lzdGVyU3dpdGNoZXIiLCJvcGVuc2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QiLCJkeW5hbWljQ29uZmlnU2VydmljZVN0YXJ0IiwiZHluYW1pY0NvbmZpZ1NlcnZpY2UiLCJnZXRTdGFydFNlcnZpY2UiLCJzdG9yZSIsImdldEFzeW5jTG9jYWxTdG9yZSIsImNsaWVudCIsImdldENsaWVudCIsImR5bmFtaWNDb25maWciLCJnZXRDb25maWciLCJwbHVnaW5Db25maWdQYXRoIiwiYXN5bmNMb2NhbFN0b3JhZ2VDb250ZXh0IiwiY29yZVN0YXJ0IiwiZ2V0U3RhcnRTZXJ2aWNlcyIsInNhdmVkT2JqZWN0c0NsaWVudCIsImdldFNjb3BlZENsaWVudCIsInVpU2V0dGluZ3NDbGllbnQiLCJ1aVNldHRpbmdzIiwiYXNTY29wZWRUb0NsaWVudCIsImlzQXNzaXN0YW50RW5hYmxlZEJ5U2V0dGluZyIsIkJvb2xlYW4iLCJFTkFCTEVfQUlfRkVBVFVSRVMiLCJjYXRjaCIsImFzc2lzdGFudCIsImNoYXRFbmFibGVkIiwiY2hhdCIsImVycm9yIiwicmVnaXN0ZXJEYXRhMlN1bW1hcnlSb3V0ZXMiLCJyZWdpc3Rlck1lc3NhZ2VQYXJzZXIiLCJtZXNzYWdlUGFyc2VyIiwiZmluZEl0ZW0iLCJmaW5kIiwiaXRlbSIsImlkIiwiRXJyb3IiLCJwdXNoIiwiQmFzaWNJbnB1dE91dHB1dFBhcnNlciIsIlZpc3VhbGl6YXRpb25DYXJkUGFyc2VyIiwicmVtb3ZlTWVzc2FnZVBhcnNlciIsInBhcnNlcklkIiwiZmluZEluZGV4Iiwic3BsaWNlIiwic3RhcnQiLCJzdG9wIiwiZXhwb3J0cyJdLCJzb3VyY2VzIjpbInBsdWdpbi50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7IGZpcnN0IH0gZnJvbSAncnhqcy9vcGVyYXRvcnMnO1xuaW1wb3J0IHsgQ29uZmlnU2NoZW1hIH0gZnJvbSAnLi4vY29tbW9uL3R5cGVzL2NvbmZpZyc7XG5pbXBvcnQge1xuICBDb3JlU2V0dXAsXG4gIENvcmVTdGFydCxcbiAgTG9nZ2VyLFxuICBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gIFBsdWdpbixcbiAgUGx1Z2luSW5pdGlhbGl6ZXJDb250ZXh0LFxufSBmcm9tICcuLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXInO1xuaW1wb3J0IHsgQXNzaXN0YW50UGx1Z2luU2V0dXAsIEFzc2lzdGFudFBsdWdpblN0YXJ0LCBNZXNzYWdlUGFyc2VyIH0gZnJvbSAnLi90eXBlcyc7XG5pbXBvcnQgeyBCYXNpY0lucHV0T3V0cHV0UGFyc2VyIH0gZnJvbSAnLi9wYXJzZXJzL2Jhc2ljX2lucHV0X291dHB1dF9wYXJzZXInO1xuaW1wb3J0IHsgVmlzdWFsaXphdGlvbkNhcmRQYXJzZXIgfSBmcm9tICcuL3BhcnNlcnMvdmlzdWFsaXphdGlvbl9jYXJkX3BhcnNlcic7XG5pbXBvcnQgeyByZWdpc3RlckNoYXRSb3V0ZXMgfSBmcm9tICcuL3JvdXRlcy9jaGF0X3JvdXRlcyc7XG5pbXBvcnQgeyByZWdpc3RlclRleHQyVml6Um91dGVzIH0gZnJvbSAnLi9yb3V0ZXMvdGV4dDJ2aXpfcm91dGVzJztcbmltcG9ydCB7IEFzc2lzdGFudFNlcnZpY2UgfSBmcm9tICcuL3NlcnZpY2VzL2Fzc2lzdGFudF9zZXJ2aWNlJztcbmltcG9ydCB7IHJlZ2lzdGVyQWdlbnRSb3V0ZXMgfSBmcm9tICcuL3JvdXRlcy9hZ2VudF9yb3V0ZXMnO1xuaW1wb3J0IHtcbiAgcmVnaXN0ZXJTdW1tYXJ5QXNzaXN0YW50Um91dGVzLFxuICByZWdpc3RlckRhdGEyU3VtbWFyeVJvdXRlcyxcbn0gZnJvbSAnLi9yb3V0ZXMvc3VtbWFyeV9yb3V0ZXMnO1xuaW1wb3J0IHsgY2FwYWJpbGl0aWVzUHJvdmlkZXIgYXMgdmlzTkxRQ2FwYWJpbGl0aWVzUHJvdmlkZXIgfSBmcm9tICcuL3Zpc190eXBlX25scS9jYXBhYmlsaXRpZXNfcHJvdmlkZXInO1xuaW1wb3J0IHsgdmlzTkxRU2F2ZWRPYmplY3RUeXBlIH0gZnJvbSAnLi92aXNfdHlwZV9ubHEvc2F2ZWRfb2JqZWN0X3R5cGUnO1xuaW1wb3J0IHsgY2FwYWJpbGl0aWVzUHJvdmlkZXIgfSBmcm9tICcuL2NhcGFiaWxpdGllcyc7XG5pbXBvcnQgeyBFTkFCTEVfQUlfRkVBVFVSRVMgfSBmcm9tICcuL3V0aWxzL2NvbnN0YW50cyc7XG5cbmV4cG9ydCBjbGFzcyBBc3Npc3RhbnRQbHVnaW4gaW1wbGVtZW50cyBQbHVnaW48QXNzaXN0YW50UGx1Z2luU2V0dXAsIEFzc2lzdGFudFBsdWdpblN0YXJ0PiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgbG9nZ2VyOiBMb2dnZXI7XG4gIHByaXZhdGUgbWVzc2FnZVBhcnNlcnM6IE1lc3NhZ2VQYXJzZXJbXSA9IFtdO1xuICBwcml2YXRlIGFzc2lzdGFudFNlcnZpY2UgPSBuZXcgQXNzaXN0YW50U2VydmljZSgpO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgaW5pdGlhbGl6ZXJDb250ZXh0OiBQbHVnaW5Jbml0aWFsaXplckNvbnRleHQpIHtcbiAgICB0aGlzLmxvZ2dlciA9IGluaXRpYWxpemVyQ29udGV4dC5sb2dnZXIuZ2V0KCk7XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgc2V0dXAoY29yZTogQ29yZVNldHVwKTogUHJvbWlzZTxBc3Npc3RhbnRQbHVnaW5TZXR1cD4ge1xuICAgIHRoaXMubG9nZ2VyLmRlYnVnKCdBc3Npc3RhbnQ6IFNldHVwJyk7XG4gICAgY29uc3QgY29uZmlnID0gYXdhaXQgdGhpcy5pbml0aWFsaXplckNvbnRleHQuY29uZmlnXG4gICAgICAuY3JlYXRlPENvbmZpZ1NjaGVtYT4oKVxuICAgICAgLnBpcGUoZmlyc3QoKSlcbiAgICAgIC50b1Byb21pc2UoKTtcblxuICAgIGNvbnN0IGFzc2lzdGFudFNlcnZpY2VTZXR1cCA9IHRoaXMuYXNzaXN0YW50U2VydmljZS5zZXR1cCgpO1xuXG4gICAgY29uc3Qgcm91dGVyID0gY29yZS5odHRwLmNyZWF0ZVJvdXRlcigpO1xuXG4gICAgY29yZS5odHRwLnJlZ2lzdGVyUm91dGVIYW5kbGVyQ29udGV4dCgnYXNzaXN0YW50X3BsdWdpbicsICgpID0+IHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIGNvbmZpZyxcbiAgICAgICAgbG9nZ2VyOiB0aGlzLmxvZ2dlcixcbiAgICAgIH07XG4gICAgfSk7XG5cbiAgICByZWdpc3RlckFnZW50Um91dGVzKHJvdXRlciwgYXNzaXN0YW50U2VydmljZVNldHVwKTtcblxuICAgIC8vIFJlZ2lzdGVyIHNlcnZlciBzaWRlIEFQSXNcbiAgICByZWdpc3RlckNoYXRSb3V0ZXMocm91dGVyLCB7XG4gICAgICBtZXNzYWdlUGFyc2VyczogdGhpcy5tZXNzYWdlUGFyc2VycyxcbiAgICAgIGF1dGg6IGNvcmUuaHR0cC5hdXRoLFxuICAgIH0pO1xuXG4gICAgLy8gUmVnaXN0ZXIgcm91dGVyIGZvciB0ZXh0IHRvIHZpc3VhbGl6YXRpb25cbiAgICBpZiAoY29uZmlnLnRleHQydml6LmVuYWJsZWQpIHtcbiAgICAgIHJlZ2lzdGVyVGV4dDJWaXpSb3V0ZXMocm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlU2V0dXApO1xuICAgICAgY29yZS5jYXBhYmlsaXRpZXMucmVnaXN0ZXJQcm92aWRlcih2aXNOTFFDYXBhYmlsaXRpZXNQcm92aWRlcik7XG4gICAgICBjb3JlLnNhdmVkT2JqZWN0cy5yZWdpc3RlclR5cGUodmlzTkxRU2F2ZWRPYmplY3RUeXBlKTtcbiAgICB9XG5cbiAgICAvLyBSZWdpc3RlciByb3V0ZXIgZm9yIGFsZXJ0IGluc2lnaHRcbiAgICBpZiAoY29uZmlnLmFsZXJ0SW5zaWdodC5lbmFibGVkKSB7XG4gICAgICByZWdpc3RlclN1bW1hcnlBc3Npc3RhbnRSb3V0ZXMocm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlU2V0dXApO1xuICAgIH1cblxuICAgIGNvcmUuY2FwYWJpbGl0aWVzLnJlZ2lzdGVyUHJvdmlkZXIoY2FwYWJpbGl0aWVzUHJvdmlkZXIpO1xuICAgIC8vIHJlZ2lzdGVyIFVJIGNhcGFiaWxpdGllcyBmcm9tIGR5bmFtaWMgY29uZmlnIHNlcnZpY2VcbiAgICBjb3JlLmNhcGFiaWxpdGllcy5yZWdpc3RlclN3aXRjaGVyKFxuICAgICAgYXN5bmMgKG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0KSA9PiB7XG4gICAgICAgIGNvbnN0IGR5bmFtaWNDb25maWdTZXJ2aWNlU3RhcnQgPSBhd2FpdCBjb3JlLmR5bmFtaWNDb25maWdTZXJ2aWNlLmdldFN0YXJ0U2VydmljZSgpO1xuICAgICAgICBjb25zdCBzdG9yZSA9IGR5bmFtaWNDb25maWdTZXJ2aWNlU3RhcnQuZ2V0QXN5bmNMb2NhbFN0b3JlKCk7XG4gICAgICAgIGNvbnN0IGNsaWVudCA9IGR5bmFtaWNDb25maWdTZXJ2aWNlU3RhcnQuZ2V0Q2xpZW50KCk7XG5cbiAgICAgICAgdHJ5IHtcbiAgICAgICAgICBjb25zdCBkeW5hbWljQ29uZmlnID0gYXdhaXQgY2xpZW50LmdldENvbmZpZyhcbiAgICAgICAgICAgIHsgcGx1Z2luQ29uZmlnUGF0aDogJ2Fzc2lzdGFudCcgfSxcbiAgICAgICAgICAgIHsgYXN5bmNMb2NhbFN0b3JhZ2VDb250ZXh0OiBzdG9yZSEgfVxuICAgICAgICAgICk7XG5cbiAgICAgICAgICBjb25zdCBbY29yZVN0YXJ0XSA9IGF3YWl0IGNvcmUuZ2V0U3RhcnRTZXJ2aWNlcygpO1xuICAgICAgICAgIGNvbnN0IHNhdmVkT2JqZWN0c0NsaWVudCA9IGNvcmVTdGFydC5zYXZlZE9iamVjdHMuZ2V0U2NvcGVkQ2xpZW50KFxuICAgICAgICAgICAgb3BlbnNlYXJjaERhc2hib2FyZHNSZXF1ZXN0XG4gICAgICAgICAgKTtcbiAgICAgICAgICBjb25zdCB1aVNldHRpbmdzQ2xpZW50ID0gY29yZVN0YXJ0LnVpU2V0dGluZ3MuYXNTY29wZWRUb0NsaWVudChzYXZlZE9iamVjdHNDbGllbnQpO1xuICAgICAgICAgIGNvbnN0IGlzQXNzaXN0YW50RW5hYmxlZEJ5U2V0dGluZyA9IEJvb2xlYW4oXG4gICAgICAgICAgICBhd2FpdCB1aVNldHRpbmdzQ2xpZW50LmdldChFTkFCTEVfQUlfRkVBVFVSRVMpLmNhdGNoKCgpID0+IGZhbHNlKVxuICAgICAgICAgICk7XG5cbiAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgYXNzaXN0YW50OiB7XG4gICAgICAgICAgICAgIGVuYWJsZWQ6IGR5bmFtaWNDb25maWcuZW5hYmxlZCAmJiBpc0Fzc2lzdGFudEVuYWJsZWRCeVNldHRpbmcsXG4gICAgICAgICAgICAgIGNoYXRFbmFibGVkOiBkeW5hbWljQ29uZmlnLmNoYXQuZW5hYmxlZCAmJiBpc0Fzc2lzdGFudEVuYWJsZWRCeVNldHRpbmcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH07XG4gICAgICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgICAgICB0aGlzLmxvZ2dlci5lcnJvcihlKTtcbiAgICAgICAgICByZXR1cm4ge307XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICApO1xuXG4gICAgLy8gUmVnaXN0ZXIgcm91dGVyIGZvciBkaXNjb3Zlcnkgc3VtbWFyeVxuICAgIHJlZ2lzdGVyRGF0YTJTdW1tYXJ5Um91dGVzKHJvdXRlciwgYXNzaXN0YW50U2VydmljZVNldHVwKTtcblxuICAgIGNvbnN0IHJlZ2lzdGVyTWVzc2FnZVBhcnNlciA9IChtZXNzYWdlUGFyc2VyOiBNZXNzYWdlUGFyc2VyKSA9PiB7XG4gICAgICBjb25zdCBmaW5kSXRlbSA9IHRoaXMubWVzc2FnZVBhcnNlcnMuZmluZCgoaXRlbSkgPT4gaXRlbS5pZCA9PT0gbWVzc2FnZVBhcnNlci5pZCk7XG4gICAgICBpZiAoZmluZEl0ZW0pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGVyZSBpcyBhbHJlYWR5IGEgbWVzc2FnZVBhcnNlciB3aG9zZSBpZCBpcyAke21lc3NhZ2VQYXJzZXIuaWR9YCk7XG4gICAgICB9XG5cbiAgICAgIHRoaXMubWVzc2FnZVBhcnNlcnMucHVzaChtZXNzYWdlUGFyc2VyKTtcbiAgICB9O1xuXG4gICAgcmVnaXN0ZXJNZXNzYWdlUGFyc2VyKEJhc2ljSW5wdXRPdXRwdXRQYXJzZXIpO1xuICAgIHJlZ2lzdGVyTWVzc2FnZVBhcnNlcihWaXN1YWxpemF0aW9uQ2FyZFBhcnNlcik7XG5cbiAgICByZXR1cm4ge1xuICAgICAgYXNzaXN0YW50U2VydmljZTogYXNzaXN0YW50U2VydmljZVNldHVwLFxuICAgICAgcmVnaXN0ZXJNZXNzYWdlUGFyc2VyLFxuICAgICAgcmVtb3ZlTWVzc2FnZVBhcnNlcjogKHBhcnNlcklkOiBNZXNzYWdlUGFyc2VyWydpZCddKSA9PiB7XG4gICAgICAgIGNvbnN0IGZpbmRJbmRleCA9IHRoaXMubWVzc2FnZVBhcnNlcnMuZmluZEluZGV4KChpdGVtKSA9PiBpdGVtLmlkID09PSBwYXJzZXJJZCk7XG4gICAgICAgIGlmIChmaW5kSW5kZXggPCAwKSB7XG4gICAgICAgICAgdGhpcy5sb2dnZXIuZXJyb3IoYFRoZXJlIGlzIG5vdCBhIG1lc3NhZ2VQYXJzZXIgd2hvc2UgaWQgaXMgJHtwYXJzZXJJZH1gKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMubWVzc2FnZVBhcnNlcnMuc3BsaWNlKGZpbmRJbmRleCwgMSk7XG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwdWJsaWMgc3RhcnQoY29yZTogQ29yZVN0YXJ0KSB7XG4gICAgdGhpcy5sb2dnZXIuZGVidWcoJ0Fzc2lzdGFudDogU3RhcnRlZCcpO1xuICAgIHRoaXMuYXNzaXN0YW50U2VydmljZS5zdGFydCgpO1xuICAgIHJldHVybiB7fTtcbiAgfVxuXG4gIHB1YmxpYyBzdG9wKCkge1xuICAgIHRoaXMuYXNzaXN0YW50U2VydmljZS5zdG9wKCk7XG4gIH1cbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBQUEsVUFBQSxHQUFBQyxPQUFBO0FBV0EsSUFBQUMsMEJBQUEsR0FBQUQsT0FBQTtBQUNBLElBQUFFLDBCQUFBLEdBQUFGLE9BQUE7QUFDQSxJQUFBRyxZQUFBLEdBQUFILE9BQUE7QUFDQSxJQUFBSSxnQkFBQSxHQUFBSixPQUFBO0FBQ0EsSUFBQUssa0JBQUEsR0FBQUwsT0FBQTtBQUNBLElBQUFNLGFBQUEsR0FBQU4sT0FBQTtBQUNBLElBQUFPLGVBQUEsR0FBQVAsT0FBQTtBQUlBLElBQUFRLHNCQUFBLEdBQUFSLE9BQUE7QUFDQSxJQUFBUyxrQkFBQSxHQUFBVCxPQUFBO0FBQ0EsSUFBQVUsYUFBQSxHQUFBVixPQUFBO0FBQ0EsSUFBQVcsVUFBQSxHQUFBWCxPQUFBO0FBQXVELFNBQUFZLGdCQUFBQyxDQUFBLEVBQUFDLENBQUEsRUFBQUMsQ0FBQSxZQUFBRCxDQUFBLEdBQUFFLGNBQUEsQ0FBQUYsQ0FBQSxNQUFBRCxDQUFBLEdBQUFJLE1BQUEsQ0FBQUMsY0FBQSxDQUFBTCxDQUFBLEVBQUFDLENBQUEsSUFBQUssS0FBQSxFQUFBSixDQUFBLEVBQUFLLFVBQUEsTUFBQUMsWUFBQSxNQUFBQyxRQUFBLFVBQUFULENBQUEsQ0FBQUMsQ0FBQSxJQUFBQyxDQUFBLEVBQUFGLENBQUE7QUFBQSxTQUFBRyxlQUFBRCxDQUFBLFFBQUFRLENBQUEsR0FBQUMsWUFBQSxDQUFBVCxDQUFBLHVDQUFBUSxDQUFBLEdBQUFBLENBQUEsR0FBQUEsQ0FBQTtBQUFBLFNBQUFDLGFBQUFULENBQUEsRUFBQUQsQ0FBQSwyQkFBQUMsQ0FBQSxLQUFBQSxDQUFBLFNBQUFBLENBQUEsTUFBQUYsQ0FBQSxHQUFBRSxDQUFBLENBQUFVLE1BQUEsQ0FBQUMsV0FBQSxrQkFBQWIsQ0FBQSxRQUFBVSxDQUFBLEdBQUFWLENBQUEsQ0FBQWMsSUFBQSxDQUFBWixDQUFBLEVBQUFELENBQUEsdUNBQUFTLENBQUEsU0FBQUEsQ0FBQSxZQUFBSyxTQUFBLHlFQUFBZCxDQUFBLEdBQUFlLE1BQUEsR0FBQUMsTUFBQSxFQUFBZixDQUFBLEtBN0J2RDtBQUNBO0FBQ0E7QUFDQTtBQTRCTyxNQUFNZ0IsZUFBZSxDQUErRDtFQUt6RkMsV0FBV0EsQ0FBa0JDLGtCQUE0QyxFQUFFO0lBQUEsS0FBOUNBLGtCQUE0QyxHQUE1Q0Esa0JBQTRDO0lBQUFyQixlQUFBO0lBQUFBLGVBQUEseUJBSC9CLEVBQUU7SUFBQUEsZUFBQSwyQkFDakIsSUFBSXNCLG1DQUFnQixDQUFDLENBQUM7SUFHL0MsSUFBSSxDQUFDQyxNQUFNLEdBQUdGLGtCQUFrQixDQUFDRSxNQUFNLENBQUNDLEdBQUcsQ0FBQyxDQUFDO0VBQy9DO0VBRUEsTUFBYUMsS0FBS0EsQ0FBQ0MsSUFBZSxFQUFpQztJQUNqRSxJQUFJLENBQUNILE1BQU0sQ0FBQ0ksS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3JDLE1BQU1DLE1BQU0sR0FBRyxNQUFNLElBQUksQ0FBQ1Asa0JBQWtCLENBQUNPLE1BQU0sQ0FDaERDLE1BQU0sQ0FBZSxDQUFDLENBQ3RCQyxJQUFJLENBQUMsSUFBQUMsZ0JBQUssRUFBQyxDQUFDLENBQUMsQ0FDYkMsU0FBUyxDQUFDLENBQUM7SUFFZCxNQUFNQyxxQkFBcUIsR0FBRyxJQUFJLENBQUNDLGdCQUFnQixDQUFDVCxLQUFLLENBQUMsQ0FBQztJQUUzRCxNQUFNVSxNQUFNLEdBQUdULElBQUksQ0FBQ1UsSUFBSSxDQUFDQyxZQUFZLENBQUMsQ0FBQztJQUV2Q1gsSUFBSSxDQUFDVSxJQUFJLENBQUNFLDJCQUEyQixDQUFDLGtCQUFrQixFQUFFLE1BQU07TUFDOUQsT0FBTztRQUNMVixNQUFNO1FBQ05MLE1BQU0sRUFBRSxJQUFJLENBQUNBO01BQ2YsQ0FBQztJQUNILENBQUMsQ0FBQztJQUVGLElBQUFnQixpQ0FBbUIsRUFBQ0osTUFBTSxFQUFFRixxQkFBcUIsQ0FBQzs7SUFFbEQ7SUFDQSxJQUFBTywrQkFBa0IsRUFBQ0wsTUFBTSxFQUFFO01BQ3pCTSxjQUFjLEVBQUUsSUFBSSxDQUFDQSxjQUFjO01BQ25DQyxJQUFJLEVBQUVoQixJQUFJLENBQUNVLElBQUksQ0FBQ007SUFDbEIsQ0FBQyxDQUFDOztJQUVGO0lBQ0EsSUFBSWQsTUFBTSxDQUFDZSxRQUFRLENBQUNDLE9BQU8sRUFBRTtNQUMzQixJQUFBQyx1Q0FBc0IsRUFBQ1YsTUFBTSxFQUFFRixxQkFBcUIsQ0FBQztNQUNyRFAsSUFBSSxDQUFDb0IsWUFBWSxDQUFDQyxnQkFBZ0IsQ0FBQ0MsMkNBQTBCLENBQUM7TUFDOUR0QixJQUFJLENBQUN1QixZQUFZLENBQUNDLFlBQVksQ0FBQ0Msd0NBQXFCLENBQUM7SUFDdkQ7O0lBRUE7SUFDQSxJQUFJdkIsTUFBTSxDQUFDd0IsWUFBWSxDQUFDUixPQUFPLEVBQUU7TUFDL0IsSUFBQVMsOENBQThCLEVBQUNsQixNQUFNLEVBQUVGLHFCQUFxQixDQUFDO0lBQy9EO0lBRUFQLElBQUksQ0FBQ29CLFlBQVksQ0FBQ0MsZ0JBQWdCLENBQUNPLGtDQUFvQixDQUFDO0lBQ3hEO0lBQ0E1QixJQUFJLENBQUNvQixZQUFZLENBQUNTLGdCQUFnQixDQUNoQyxNQUFPQywyQkFBd0QsSUFBSztNQUNsRSxNQUFNQyx5QkFBeUIsR0FBRyxNQUFNL0IsSUFBSSxDQUFDZ0Msb0JBQW9CLENBQUNDLGVBQWUsQ0FBQyxDQUFDO01BQ25GLE1BQU1DLEtBQUssR0FBR0gseUJBQXlCLENBQUNJLGtCQUFrQixDQUFDLENBQUM7TUFDNUQsTUFBTUMsTUFBTSxHQUFHTCx5QkFBeUIsQ0FBQ00sU0FBUyxDQUFDLENBQUM7TUFFcEQsSUFBSTtRQUNGLE1BQU1DLGFBQWEsR0FBRyxNQUFNRixNQUFNLENBQUNHLFNBQVMsQ0FDMUM7VUFBRUMsZ0JBQWdCLEVBQUU7UUFBWSxDQUFDLEVBQ2pDO1VBQUVDLHdCQUF3QixFQUFFUDtRQUFPLENBQ3JDLENBQUM7UUFFRCxNQUFNLENBQUNRLFNBQVMsQ0FBQyxHQUFHLE1BQU0xQyxJQUFJLENBQUMyQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2pELE1BQU1DLGtCQUFrQixHQUFHRixTQUFTLENBQUNuQixZQUFZLENBQUNzQixlQUFlLENBQy9EZiwyQkFDRixDQUFDO1FBQ0QsTUFBTWdCLGdCQUFnQixHQUFHSixTQUFTLENBQUNLLFVBQVUsQ0FBQ0MsZ0JBQWdCLENBQUNKLGtCQUFrQixDQUFDO1FBQ2xGLE1BQU1LLDJCQUEyQixHQUFHQyxPQUFPLENBQ3pDLE1BQU1KLGdCQUFnQixDQUFDaEQsR0FBRyxDQUFDcUQsNkJBQWtCLENBQUMsQ0FBQ0MsS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUNsRSxDQUFDO1FBRUQsT0FBTztVQUNMQyxTQUFTLEVBQUU7WUFDVG5DLE9BQU8sRUFBRW9CLGFBQWEsQ0FBQ3BCLE9BQU8sSUFBSStCLDJCQUEyQjtZQUM3REssV0FBVyxFQUFFaEIsYUFBYSxDQUFDaUIsSUFBSSxDQUFDckMsT0FBTyxJQUFJK0I7VUFDN0M7UUFDRixDQUFDO01BQ0gsQ0FBQyxDQUFDLE9BQU8xRSxDQUFDLEVBQUU7UUFDVixJQUFJLENBQUNzQixNQUFNLENBQUMyRCxLQUFLLENBQUNqRixDQUFDLENBQUM7UUFDcEIsT0FBTyxDQUFDLENBQUM7TUFDWDtJQUNGLENBQ0YsQ0FBQzs7SUFFRDtJQUNBLElBQUFrRiwwQ0FBMEIsRUFBQ2hELE1BQU0sRUFBRUYscUJBQXFCLENBQUM7SUFFekQsTUFBTW1ELHFCQUFxQixHQUFJQyxhQUE0QixJQUFLO01BQzlELE1BQU1DLFFBQVEsR0FBRyxJQUFJLENBQUM3QyxjQUFjLENBQUM4QyxJQUFJLENBQUVDLElBQUksSUFBS0EsSUFBSSxDQUFDQyxFQUFFLEtBQUtKLGFBQWEsQ0FBQ0ksRUFBRSxDQUFDO01BQ2pGLElBQUlILFFBQVEsRUFBRTtRQUNaLE1BQU0sSUFBSUksS0FBSyxDQUFFLGdEQUErQ0wsYUFBYSxDQUFDSSxFQUFHLEVBQUMsQ0FBQztNQUNyRjtNQUVBLElBQUksQ0FBQ2hELGNBQWMsQ0FBQ2tELElBQUksQ0FBQ04sYUFBYSxDQUFDO0lBQ3pDLENBQUM7SUFFREQscUJBQXFCLENBQUNRLGlEQUFzQixDQUFDO0lBQzdDUixxQkFBcUIsQ0FBQ1Msa0RBQXVCLENBQUM7SUFFOUMsT0FBTztNQUNMM0QsZ0JBQWdCLEVBQUVELHFCQUFxQjtNQUN2Q21ELHFCQUFxQjtNQUNyQlUsbUJBQW1CLEVBQUdDLFFBQTZCLElBQUs7UUFDdEQsTUFBTUMsU0FBUyxHQUFHLElBQUksQ0FBQ3ZELGNBQWMsQ0FBQ3VELFNBQVMsQ0FBRVIsSUFBSSxJQUFLQSxJQUFJLENBQUNDLEVBQUUsS0FBS00sUUFBUSxDQUFDO1FBQy9FLElBQUlDLFNBQVMsR0FBRyxDQUFDLEVBQUU7VUFDakIsSUFBSSxDQUFDekUsTUFBTSxDQUFDMkQsS0FBSyxDQUFFLDRDQUEyQ2EsUUFBUyxFQUFDLENBQUM7UUFDM0U7UUFFQSxJQUFJLENBQUN0RCxjQUFjLENBQUN3RCxNQUFNLENBQUNELFNBQVMsRUFBRSxDQUFDLENBQUM7TUFDMUM7SUFDRixDQUFDO0VBQ0g7RUFFT0UsS0FBS0EsQ0FBQ3hFLElBQWUsRUFBRTtJQUM1QixJQUFJLENBQUNILE1BQU0sQ0FBQ0ksS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBQ3ZDLElBQUksQ0FBQ08sZ0JBQWdCLENBQUNnRSxLQUFLLENBQUMsQ0FBQztJQUM3QixPQUFPLENBQUMsQ0FBQztFQUNYO0VBRU9DLElBQUlBLENBQUEsRUFBRztJQUNaLElBQUksQ0FBQ2pFLGdCQUFnQixDQUFDaUUsSUFBSSxDQUFDLENBQUM7RUFDOUI7QUFDRjtBQUFDQyxPQUFBLENBQUFqRixlQUFBLEdBQUFBLGVBQUEifQ==