/*
 * SPDX-FileCopyrightText: 2021 Daniel Vrátil <dvratil@kde.org>
 * SPDX-FileCopyrightText: 2022 Claudio Cambra <claudio.cambra@kde.org>
 *
 * SPDX-License-Identifier: LGPL-2.1-only
 * SPDX-License-Identifier: LGPL-3.0-only
 * SPDX-License-Identifier: LicenseRef-KDE-Accepted-LGPL
 */

#pragma once

#include <QSharedDataPointer>
#include "kgapipeople_export.h"

#include <QString>

#include <optional>

class QJsonObject;
class QJsonValue;

namespace KGAPI2::People
{
class ProfileMetadata;

/**
 * The source of a field.
 *
 * @see https://developers.google.com/people/api/rest/v1/people#source
 * @since 5.23.0
 **/
class KGAPIPEOPLE_EXPORT Source
{
public:
    enum class Type {
        SOURCE_TYPE_UNSPECIFIED, ///< Unspecified.
        ACCOUNT, ///< [Google Account](https://accounts.google.com).
        PROFILE, ///< [Google profile](https://profiles.google.com). You can view the profile at
                 ///< [https://profiles.google.com/](https://profiles.google.com/){id}, where {id} is the source id.
        DOMAIN_PROFILE, ///< [Google Workspace domain profile](https://support.google.com/a/answer/1628008).
        CONTACT, ///< [Google contact](https://contacts.google.com). You can view the contact at [https://contact.google.com/](https://contact.google.com/){id},
                 ///< where {id} is the source id.
        OTHER_CONTACT, ///< [Google "Other contact"](https://contacts.google.com/other).
        DOMAIN_CONTACT, ///< [Google Workspace domain shared contact](https://support.google.com/a/answer/9281635).
    };

    /** Constructs a new Source **/
    explicit Source();
    Source(const Source &);
    Source(Source &&) noexcept;
    Source &operator=(const Source &);
    Source &operator=(Source &&) noexcept;
    /** Destructor. **/
    ~Source();

    bool operator==(const Source &) const;
    bool operator!=(const Source &) const;

    [[nodiscard]] static Source fromJSON(const QJsonObject &);
    [[nodiscard]] QJsonValue toJSON() const;

    /** **Only populated in `person.metadata.sources`.** The [HTTP entity tag](https://en.wikipedia.org/wiki/HTTP_ETag) of the source. Used for web cache
     * validation. **/
    [[nodiscard]] QString etag() const;
    /** Sets value of the etag property. **/
    void setEtag(const QString &value);

    /** Output only. **Only populated in `person.metadata.sources`.** Metadata about a source of type PROFILE. **/
    [[nodiscard]] ProfileMetadata profileMetadata() const;

    /** The unique identifier within the source type generated by the server. **/
    [[nodiscard]] QString id() const;
    /** Sets value of the id property. **/
    void setId(const QString &value);

    /** Output only. **Only populated in `person.metadata.sources`.** Last update timestamp of this source. **/
    [[nodiscard]] QString updateTime() const;

    /** The source type. **/
    [[nodiscard]] Source::Type type() const;
    /** Sets value of the type property. **/
    void setType(Source::Type value);

private:
    class Private;
    QSharedDataPointer<Private> d;
}; // Source

} // namespace KGAPI2::People
