<?php

/**
 * SPDX-FileCopyrightText: 2016-2024 Nextcloud GmbH and Nextcloud contributors
 * SPDX-FileCopyrightText: 2016 ownCloud, Inc.
 * SPDX-License-Identifier: AGPL-3.0-only
 */
namespace OC\Files\Mount;

use OCP\Files\Config\IHomeMountProvider;
use OCP\Files\Storage\IStorageFactory;
use OCP\IConfig;
use OCP\IUser;
use Psr\Log\LoggerInterface;

/**
 * Mount provider for object store home storages
 */
class ObjectHomeMountProvider implements IHomeMountProvider {
	/**
	 * @var IConfig
	 */
	private $config;

	/**
	 * ObjectStoreHomeMountProvider constructor.
	 *
	 * @param IConfig $config
	 */
	public function __construct(IConfig $config) {
		$this->config = $config;
	}

	/**
	 * Get the cache mount for a user
	 *
	 * @param IUser $user
	 * @param IStorageFactory $loader
	 * @return \OCP\Files\Mount\IMountPoint
	 */
	public function getHomeMountForUser(IUser $user, IStorageFactory $loader) {
		$config = $this->getMultiBucketObjectStoreConfig($user);
		if ($config === null) {
			$config = $this->getSingleBucketObjectStoreConfig($user);
		}

		if ($config === null) {
			return null;
		}

		return new HomeMountPoint($user, '\OC\Files\ObjectStore\HomeObjectStoreStorage', '/' . $user->getUID(), $config['arguments'], $loader, null, null, self::class);
	}

	/**
	 * @param IUser $user
	 * @return array|null
	 */
	private function getSingleBucketObjectStoreConfig(IUser $user) {
		$config = $this->config->getSystemValue('objectstore');
		if (!is_array($config)) {
			return null;
		}

		// sanity checks
		if (empty($config['class'])) {
			\OC::$server->get(LoggerInterface::class)->error('No class given for objectstore', ['app' => 'files']);
		}
		if (!isset($config['arguments'])) {
			$config['arguments'] = [];
		}
		// instantiate object store implementation
		$config['arguments']['objectstore'] = new $config['class']($config['arguments']);

		$config['arguments']['user'] = $user;

		return $config;
	}

	/**
	 * @param IUser $user
	 * @return array|null
	 */
	private function getMultiBucketObjectStoreConfig(IUser $user) {
		$config = $this->config->getSystemValue('objectstore_multibucket');
		if (!is_array($config)) {
			return null;
		}

		// sanity checks
		if (empty($config['class'])) {
			\OC::$server->get(LoggerInterface::class)->error('No class given for objectstore', ['app' => 'files']);
		}
		if (!isset($config['arguments'])) {
			$config['arguments'] = [];
		}

		$bucket = $this->config->getUserValue($user->getUID(), 'homeobjectstore', 'bucket', null);

		if ($bucket === null) {
			/*
			 * Use any provided bucket argument as prefix
			 * and add the mapping from username => bucket
			 */
			if (!isset($config['arguments']['bucket'])) {
				$config['arguments']['bucket'] = '';
			}
			$mapper = new \OC\Files\ObjectStore\Mapper($user, $this->config);
			$numBuckets = $config['arguments']['num_buckets'] ?? 64;
			$config['arguments']['bucket'] .= $mapper->getBucket($numBuckets);

			$this->config->setUserValue($user->getUID(), 'homeobjectstore', 'bucket', $config['arguments']['bucket']);
		} else {
			$config['arguments']['bucket'] = $bucket;
		}

		// instantiate object store implementation
		$config['arguments']['objectstore'] = new $config['class']($config['arguments']);

		$config['arguments']['user'] = $user;

		return $config;
	}
}
