use core:lang;
use lang:bnf;
use lang:bs;
use lang:bs:macro;

/**
 * A declared parser.
 */
class Parser extends lang:bs:NamedDecl {
	// Override to create function body.
	FnBody createBody(BSRawFn fn, Bool binary, Scope scope, Rule start, NameSet[] include) : abstract;

	// Create.
	init(Scope scope, SStr name) {
		init {
			scope = scope;
			name = name;
		}
	}

	// Options for this function.
	private SFreeOptions? options;

	// Scope.
	private Scope scope;

	// Name.
	SStr name;

	// Syntax entered in the parser block.
	ParserSyntax? syntax;

	// Parse a binary buffer rather than a string.
	private Bool binary;

	// Create two parsers from the same grammar: one for strings and one for binary blobs.
	private Bool dual;

	// Called to make the parser into a binary parser.
	void makeBinary() {
		binary = true;
	}

	// Called to make the parser into a "dual" parser.
	void makeDual() {
		dual = true;
	}

	// Generate the overloads for this parser.
	MultiDecl parserOverloads() {
		if (dual & binary)
			throw SyntaxError(name.pos, "Can not use both 'binary' and 'text and binary' for the same parser.");

		TObject[] overloads;
		// Add ourselves.
		overloads << this << ParserOverload(this, binary);
		// If we were asked to be a dual parser, add that as well.
		if (dual) {
			BinaryWrapper binary(this);
			overloads << binary << ParserOverload(binary, true);
		}
		return MultiDecl(overloads);
	}

	// Set syntax. Called by the grammar.
	void setSyntax(ParserSyntax syntax) {
		this.syntax = syntax;
	}

	// Set options. Called by the grammar.
	void setOptions(SFreeOptions options) {
		this.options = options;
	}

	// Function class so that we can do lazy-loading to get load order right.
	class ParserFn extends BSRawFn {
		init(Parser parser, Bool binary, ParserSyntax syntax, SrcName startName, NameLookup scopeTop) {
			Package syntaxPkg = syntax.createPackage(parser.scope, scopeTop);

			// Add includes. Most importantly so that we may access both 'syntaxPkg' and the current package.
			// We add the new package as the topmost, so that things there take precedence.
			NameSet[] include;
			include << ScopeLookup:firstPkg(scopeTop);
			for (name in syntax.use) {
				unless (inc = parser.scope.find(name) as NameSet)
					throw SyntaxError(name.pos, "Unknown package: ${name}");
				include << inc;
			}

			// Also add core.io, otherwise we can't access core:io:cut (which is a non-member):
			if (binary)
				include << named{core:io};

			// Note that we change the top to the child package. That way things there take precedence.
			Scope s = addIncludes(parser.scope, include).child(syntaxPkg);
			Rule start = findRule(s, startName);

			// We also need the syntax package in 'includes', otherwise we don't include the grammar for it.
			include << syntaxPkg;

			// Now we can start creating the actual parser functions!
			SimpleName resultName;
			resultName.add("parser");

			ValParam[] params = start.params.clone;
			if (binary) {
				params.insert(0, ValParam(named{core:io:Buffer}, " input"));
				params.insert(1, ValParam(named{Nat}, " start"));
				resultName.add("BinaryResult", [start.result]);
			} else {
				params.insert(0, ValParam(named{Str}, " input"));
				params.insert(1, ValParam(named{Str:Iter}, " start"));
				resultName.add("Result", [start.result]);
			}

			var resultType = s.value(resultName, parser.name.pos);

			init(resultType, parser.name, params, null) {
				parser = parser;
				binary = binary;
				start = start;
				scope = s;
				include = include;
			}
		}

		// Data we need when creating the body.
		Parser? parser;
		Rule? start;
		Scope scope;
		NameSet[] include;
		Bool binary;

		FnBody createBody() : override {
			unless (parser)
				throw InternalError("Should not need to create the body more than once!");
			unless (start)
				throw InternalError("Should not need to create the body more than once!");

			parser.createBody(this, binary, scope.withPos(parser.name.pos), start, include);
		}

		void clearBody() : override {
			// Shed some weight.
			parser = null;
			start = null;
			scope = Scope();
			include.clear();
		}
	}

	// Create the parser, but provide 'binary' explicitly.
	private ParserFn createExplicit(Bool binary) {
		unless (syntax)
			throw InternalError("The syntax for the parser was not set properly.");
		unless (startName = syntax.start)
			throw SyntaxError(name.pos, "You need to specify the start rule with 'start = <name>;'.");
		unless (scopeTop = scope.top)
			throw InternalError("Expected to find a package in the scope.");

		ParserFn fn(this, binary, syntax, startName, scopeTop);

		// Apply options.
		if (thread)
			fn.thread(scope, thread);
		if (options)
			options.transform(scope, fn);

		fn;
	}

	// Create the parser. The generated function takes (Str, Str:Iter, ...) as parameters, where
	// ... is the parameters to the root rule.
	protected Named doCreate() : override {
		return createExplicit(binary);
	}


	/**
	 * Wrapper of a parser that produces a binary of the other parser.
	 */
	private class BinaryWrapper extends lang:bs:NamedDecl {
		// Wrapped parser.
		Parser parser;

		// Create.
		init(Parser parser) {
			init { parser = parser; }
		}

		// Creation.
		protected Named doCreate() : override {
			return parser.createExplicit(true);
		}
	}
}

// Find a rule. Throws on error.
private Rule findRule(Scope scope, SrcName name) on Compiler {
	if (rule = scope.find(name) as Rule)
		rule;
	else
		throw SyntaxError(name.pos, "Unknown rule: ${name}");
}

// Add a set of NameSet:s to a Scope. Making sure not to modify the original scope.
private Scope addIncludes(Scope scope, NameSet[] add) on Compiler {
	if (lookup = scope.lookup) {
		lookup = lookup.clone;
		if (lookup as BSLookup) {
			for (x in add)
				if (x as Package)
					lookup.addInclude(x);
		} else if (lookup as ScopeExtra) {
			for (x in add)
				lookup.addExtra(x);
		}
		scope.lookup = lookup;
	}
	scope;
}


/**
 * Overload of a parser. Wraps the X(Str, Iter) with X(Str).
 */
class ParserOverload extends lang:bs:NamedDecl {
	init(lang:bs:NamedDecl parser, Bool binary) {
		init {
			wrap = parser;
			binary = binary;
		}
	}

	lang:bs:NamedDecl wrap;
	Bool binary;

	protected Named doCreate() : override {
		unless (wrapFn = wrap.create() as BSRawFn)
			throw InternalError("Expected a function.");

		ValParam[] params;
		for (i, x in wrapFn.docParams) {
			if (i != 1) // Remove the position parameter
				params << ValParam(x.type, x.name);
		}

		BSTreeFn fn(wrapFn.result, SStr(wrapFn.name, wrapFn.pos), params, wrapFn.declaredThread);

		// Apply thread options.
		if (thread = wrapFn.declaredThread)
			fn.runOn(thread);

		fn;
	}

	protected void doResolve(Named fn) : override {
		unless (wrapFn = wrap.create() as BSRawFn)
			throw InternalError("Expected a function.");

		unless (fn as BSTreeFn)
			throw InternalError("Expected a tree function.");

		Scope scope(wrapFn);

		FnBody body(fn, scope);
		fn.body = body;

		Actuals actuals;
		LocalVarAccess str(SrcPos(), body.parameters[0]); // Source string.
		actuals.add(str);
		if (binary)
			actuals.add(NumLiteral(SrcPos(), 0));
		else
			actuals.add(FnCall(SrcPos(), scope, named{Str:begin<Str>}, Actuals(str)));

		for (Nat i = 2; i < body.parameters.count; i++)
			actuals.add(LocalVarAccess(SrcPos(), body.parameters[i]));
		body.add(FnCall(SrcPos(), scope, wrapFn, actuals));
	}
}

class ParserSyntax extends lang:bnf:FileContents {
	// Name of the start rule.
	SrcName? start;

	void push(FileItem item) : override {
		if (item as StartDecl) {
			if (start)
				throw SyntaxError(item.name.pos, "You can only specify the start rule once.");
			start = item.name;
		} else {
			super:push(item);
		}
	}
}

// Create a package that contains the productions in here.
Package createPackage(lang:bnf:FileContents this, Scope scope, NameLookup parent) on Compiler {
	Package p("<anonymous>");
	p.parentLookup = parent;

	SyntaxLookup lookup;
	if (old = scope.lookup as ScopeExtra) {
		lookup.addExtra(parent);
		for (x in old.extra)
			lookup.addExtra(x, false);
	} else if (old = scope.lookup as BSLookup) {
		lookup.addExtra(parent);
		for (x in old.includes)
			lookup.addExtra(x, false);
	}

	// New includes.
	for (x in use) {
		unless (found = rootScope.find(x))
			throw SyntaxError(x.pos, "The package ${x} does not exist!");
		lookup.addExtra(found);
	}

	scope.lookup = lookup;
	scope.top = p;

	for (x in rules) {
		p.add(Rule(x, scope));
	}

	var delims = delimiters(scope);

	for (x in productions) {
		Str name = if (n = x.name) {
			n;
		} else {
			p.anonName();
		};
		p.add(ProductionType(name, x, delims, scope));
	}

	p;
}

// Declare what the start rule is.
class StartDecl extends lang:bnf:FileItem {
	init(SrcName name) {
		init { name = name; }
	}

	SrcName name;
}
