/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/***************************************************************************
 * Copyright (C) 2017-2025 ZmartZone Holding BV
 * Copyright (C) 2013-2017 Ping Identity Corporation
 * All rights reserved.
 *
 * DISCLAIMER OF WARRANTIES:
 *
 * THE SOFTWARE PROVIDED HEREUNDER IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
 * ANY WARRANTIES OR REPRESENTATIONS EXPRESS, IMPLIED OR STATUTORY; INCLUDING,
 * WITHOUT LIMITATION, WARRANTIES OF QUALITY, PERFORMANCE, NONINFRINGEMENT,
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  NOR ARE THERE ANY
 * WARRANTIES CREATED BY A COURSE OR DEALING, COURSE OF PERFORMANCE OR TRADE
 * USAGE.  FURTHERMORE, THERE ARE NO WARRANTIES THAT THE SOFTWARE WILL MEET
 * YOUR NEEDS OR BE FREE FROM ERRORS, OR THAT THE OPERATION OF THE SOFTWARE
 * WILL BE UNINTERRUPTED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * core cache functions: locking, crypto and utils
 *
 * @Author: Hans Zandbelt - hans.zandbelt@openidc.com
 */

#ifndef WIN32
#include <unistd.h>
#endif

#include "cache/cache.h"
#include "cfg/cache.h"
#include "cfg/cfg_int.h"
#include "jose.h"
#include "metrics.h"
#include "util/util.h"

#ifdef AP_NEED_SET_MUTEX_PERMS
#include "unixd.h"
#endif

/* create the cache lock context */
oidc_cache_mutex_t *oidc_cache_mutex_create(apr_pool_t *pool, apr_byte_t global) {
	oidc_cache_mutex_t *ctx = apr_pcalloc(pool, sizeof(oidc_cache_mutex_t));
	ctx->gmutex = NULL;
	ctx->pmutex = NULL;
	ctx->mutex_filename = NULL;
	ctx->is_parent = TRUE;
	ctx->is_global = global;
	return ctx;
}

#define OIDC_CACHE_ERROR_STR_MAX 255

/*
 * convert a apr status code to a string
 */
char *oidc_cache_status2str(apr_pool_t *p, apr_status_t statcode) {
	char buf[OIDC_CACHE_ERROR_STR_MAX];
	apr_strerror(statcode, buf, OIDC_CACHE_ERROR_STR_MAX);
	return apr_pstrdup(p, buf);
}

apr_byte_t oidc_cache_mutex_post_config(server_rec *s, oidc_cache_mutex_t *m, const char *type) {

	apr_status_t rv = APR_SUCCESS;
	const char *dir;

	/* construct the mutex filename */
	rv = apr_temp_dir_get(&dir, s->process->pool);
	if (rv != APR_SUCCESS) {
		oidc_serror(s, "apr_temp_dir_get failed: could not find a temp dir: %s",
			    oidc_cache_status2str(s->process->pool, rv));
		return FALSE;
	}

	m->mutex_filename =
	    apr_psprintf(s->process->pool, "%s/mod_auth_openidc_%s_mutex.%ld.%pp", dir, type, (long int)getpid(), s);

	/* set the lock type */
	apr_lockmech_e mech =
#ifdef OIDC_LOCK
	    OIDC_LOCK
#elif APR_HAS_POSIXSEM_SERIALIZE
	    APR_LOCK_POSIXSEM
#else
	    APR_LOCK_DEFAULT
#endif
	    ;

	/* create the mutex lock */
	if (m->is_global)
		rv = apr_global_mutex_create(&m->gmutex, (const char *)m->mutex_filename, mech, s->process->pool);
	else
		rv = apr_proc_mutex_create(&m->pmutex, (const char *)m->mutex_filename, mech, s->process->pool);

	if (rv != APR_SUCCESS) {
		oidc_serror(
		    s, "apr_global_mutex_create/apr_proc_mutex_create failed to create mutex (%d) on file %s: %s (%d)",
		    mech, m->mutex_filename, oidc_cache_status2str(s->process->pool, rv), rv);
		return FALSE;
	}

	/* need this on Linux */
#ifdef AP_NEED_SET_MUTEX_PERMS
	if (m->is_global) {
#if MODULE_MAGIC_NUMBER_MAJOR >= 20081201
		rv = ap_unixd_set_global_mutex_perms(m->gmutex);
#else
		rv = unixd_set_global_mutex_perms(m->gmutex);
#endif
		if (rv != APR_SUCCESS) {
			oidc_serror(s, "unixd_set_global_mutex_perms failed; could not set permissions: %s (%d)",
				    oidc_cache_status2str(s->process->pool, rv), rv);
			return FALSE;
		}
	}
#endif

	oidc_slog(s, APLOG_TRACE1, "create: %pp (m=%pp,s=%pp, p=%d)", m, m->gmutex ? m->gmutex : 0, s, m->is_parent);

	return TRUE;
}

/*
 * initialize the cache lock in a child process
 */
apr_status_t oidc_cache_mutex_child_init(apr_pool_t *p, server_rec *s, oidc_cache_mutex_t *m) {

	oidc_slog(s, APLOG_TRACE1, "init: %pp (m=%pp,s=%pp, p=%d)", m, m->gmutex ? m->gmutex : 0, s, m->is_parent);

	if (m->is_parent == FALSE)
		return APR_SUCCESS;

	/* initialize the lock for the child process */
	apr_status_t rv;

	if (m->is_global)
		rv = apr_global_mutex_child_init(&m->gmutex, (const char *)m->mutex_filename, p);
	else
		rv = apr_proc_mutex_child_init(&m->pmutex, (const char *)m->mutex_filename, p);

	if (rv != APR_SUCCESS) {
		oidc_serror(
		    s,
		    "apr_global_mutex_child_init/apr_proc_mutex_child_init failed to reopen mutex on file %s: %s (%d)",
		    m->mutex_filename, oidc_cache_status2str(p, rv), rv);
	}

	m->is_parent = FALSE;

	return rv;
}

/*
 * mutex lock
 */
apr_byte_t oidc_cache_mutex_lock(apr_pool_t *pool, server_rec *s, oidc_cache_mutex_t *m) {

	apr_status_t rv;

	if (m->is_global)
		rv = apr_global_mutex_lock(m->gmutex);
	else
		rv = apr_proc_mutex_lock(m->pmutex);

	if (rv != APR_SUCCESS)
		oidc_serror(s, "apr_global_mutex_lock/apr_proc_mutex_lock failed: %s (%d)",
			    oidc_cache_status2str(pool, rv), rv);

	return TRUE;
}

/*
 * mutex unlock
 */
apr_byte_t oidc_cache_mutex_unlock(apr_pool_t *pool, server_rec *s, oidc_cache_mutex_t *m) {

	apr_status_t rv;

	if (m->is_global)
		rv = apr_global_mutex_unlock(m->gmutex);
	else
		rv = apr_proc_mutex_unlock(m->pmutex);

	if (rv != APR_SUCCESS)
		oidc_serror(s, "apr_global_mutex_unlock/apr_proc_mutex_unlock failed: %s (%d)",
			    oidc_cache_status2str(pool, rv), rv);

	return TRUE;
}

/*
 * destroy mutex
 */
apr_byte_t oidc_cache_mutex_destroy(server_rec *s, oidc_cache_mutex_t *m) {

	apr_status_t rv = APR_SUCCESS;

	oidc_slog(s, APLOG_TRACE1, "init: %pp (m=%pp,s=%pp, p=%d)", m, m->gmutex ? m->gmutex : 0, s, m->is_parent);

	if (m && (m->is_parent == TRUE)) {
		if ((m->is_global) && (m->gmutex)) {
			rv = apr_global_mutex_destroy(m->gmutex);
			m->gmutex = NULL;
		} else if (m->pmutex) {
			rv = apr_proc_mutex_destroy(m->pmutex);
			m->pmutex = NULL;
		}
		oidc_sdebug(s, "apr_global_mutex_destroy/apr_proc_mutex_destroy returned :%d", rv);
	}

	return (rv == APR_SUCCESS);
}

/*
 * AES GCM encrypt using the crypto passphrase as symmetric key
 */
static inline apr_byte_t oidc_cache_crypto_encrypt(request_rec *r, const char *plaintext,
						   const oidc_crypto_passphrase_t *passphrase, char **result) {
	return oidc_util_jwt_create(r, passphrase, plaintext, result);
}

/*
 * AES GCM decrypt using the crypto passphrase as symmetric key
 */
static inline apr_byte_t oidc_cache_crypto_decrypt(request_rec *r, const char *cache_value, char *secret,
						   char **plaintext) {
	oidc_crypto_passphrase_t passphrase;
	passphrase.secret1 = secret;
	passphrase.secret2 = NULL;
	return oidc_util_jwt_verify(r, &passphrase, cache_value, plaintext);
}

/*
 * hash a cache key, useful for large keys e.g. JWT access/refresh tokens
 */
static inline char *oidc_cache_get_hashed_key(request_rec *r, const char *key) {
	char *output = NULL;
	if (oidc_util_hash_string_and_base64url_encode(r, OIDC_JOSE_ALG_SHA256, key, &output) == FALSE) {
		oidc_error(r, "oidc_util_hash_string_and_base64url_encode returned an error");
		return NULL;
	}
	return output;
}

/*
 * construct a cache key
 */
static inline apr_byte_t oidc_cache_get_key(request_rec *r, const char *s_key, const char *s_secret, int encrypted,
					    const char **r_key) {
	/* see if encryption is turned on, so we'll hash passphrase+key */
	if (encrypted == 1) {
		if (s_secret == NULL) {
			oidc_error(r, "could not decrypt cache entry because " OIDCCryptoPassphrase " is not set");
			return FALSE;
		}
		*r_key = oidc_cache_get_hashed_key(r, apr_psprintf(r->pool, "%s:%s", s_secret, s_key));
	} else if (_oidc_strlen(s_key) >= OIDC_CACHE_KEY_SIZE_MAX) {
		*r_key = oidc_cache_get_hashed_key(r, s_key);
	} else {
		*r_key = s_key;
	}
	return TRUE;
}

#define OIDC_CACHE_PREFIX_ENV_VAR "OIDC_CACHE_PREFIX"

/*
 * construct the cache section identifier with a optional prefix if set in an environment variable
 */
static const char *oidc_cache_section_get(request_rec *r, const char *section) {
	const char *prefix = apr_table_get(r->subprocess_env, OIDC_CACHE_PREFIX_ENV_VAR);
	return (prefix == NULL) ? section : apr_psprintf(r->pool, "%s%s", prefix, section);
}

/*
 * get a key/value string pair from the cache, possibly decrypting it
 */
apr_byte_t oidc_cache_get(request_rec *r, const char *section, const char *key, char **value) {

	oidc_cfg_t *cfg = ap_get_module_config(r->server->module_config, &auth_openidc_module);
	int encrypted = oidc_cfg_cache_encrypt_get(cfg);
	apr_byte_t rc = FALSE;
	char *msg = NULL;
	const char *s_key = NULL;
	char *cache_value = NULL;
	char *s_secret = NULL;
	const char *s_section = oidc_cache_section_get(r, section);

	oidc_debug(r, "enter: %s (section=%s, decrypt=%d, type=%s)", key, s_section, encrypted, cfg->cache.impl->name);

	s_secret = cfg->crypto_passphrase.secret1;
	if (oidc_cache_get_key(r, key, s_secret, encrypted, &s_key) == FALSE)
		goto end;

	OIDC_METRICS_TIMING_START(r, cfg);

	/* get the value from the cache */
	if (cfg->cache.impl->get(r, s_section, s_key, &cache_value) == FALSE)
		goto end;

	/* see if it is any good */
	if ((cache_value == NULL) && (encrypted == 1) && (cfg->crypto_passphrase.secret2 != NULL)) {
		oidc_debug(r, "2nd try with previous passphrase");
		s_secret = cfg->crypto_passphrase.secret2;
		if (oidc_cache_get_key(r, key, s_secret, encrypted, &s_key) == FALSE)
			goto end;
		if (cfg->cache.impl->get(r, s_section, s_key, &cache_value) == FALSE)
			goto end;
	}

	OIDC_METRICS_TIMING_ADD(r, cfg, OM_CACHE_READ);

	rc = TRUE;

	if (cache_value == NULL)
		goto end;

	/* see if encryption is turned on */
	if (encrypted == 0) {
		*value = apr_pstrdup(r->pool, cache_value);
		goto end;
	}

	rc = oidc_cache_crypto_decrypt(r, cache_value, s_secret, value);

end:

	/* log the result */
	msg = apr_psprintf(r->pool, "from %s cache backend for %skey %s", cfg->cache.impl->name,
			   encrypted ? "encrypted " : "", key);

	if (rc == TRUE) {
		if (*value != NULL)
			oidc_debug(r, "cache hit: return %d bytes %s", *value ? (int)_oidc_strlen(*value) : 0, msg);
		else
			oidc_debug(r, "cache miss %s", msg);
	} else {
		OIDC_METRICS_COUNTER_INC(r, cfg, OM_CACHE_ERROR);
		oidc_warn(r, "error retrieving value %s", msg);
	}

	return rc;
}

/*
 * store a key/value string pair in the cache, possibly in encrypted form
 */
apr_byte_t oidc_cache_set(request_rec *r, const char *section, const char *key, const char *value, apr_time_t expiry) {

	oidc_cfg_t *cfg = ap_get_module_config(r->server->module_config, &auth_openidc_module);
	int encrypted = oidc_cfg_cache_encrypt_get(cfg);
	char *encoded = NULL;
	apr_byte_t rc = FALSE;
	char *msg = NULL;
	const char *s_key = NULL;
	const char *s_section = oidc_cache_section_get(r, section);

	oidc_debug(r, "enter: %s (section=%s, len=%d, encrypt=%d, ttl(s)=%" APR_TIME_T_FMT ", type=%s)", key, s_section,
		   value ? (int)_oidc_strlen(value) : 0, encrypted, apr_time_sec(expiry - apr_time_now()),
		   cfg->cache.impl->name);

	if (oidc_cache_get_key(r, key, cfg->crypto_passphrase.secret1, encrypted, &s_key) == FALSE)
		goto end;

	/* see if we need to encrypt */
	if ((encrypted == 1) && (value != NULL)) {
		if (oidc_cache_crypto_encrypt(r, value, &cfg->crypto_passphrase, &encoded) == FALSE)
			goto end;
		value = encoded;
	}

	OIDC_METRICS_TIMING_START(r, cfg);

	/* store the resulting value in the cache */
	rc = cfg->cache.impl->set(r, s_section, s_key, value, expiry);

	OIDC_METRICS_TIMING_ADD(r, cfg, OM_CACHE_WRITE);

end:

	/* log the result */
	msg = apr_psprintf(r->pool, "%d bytes in %s cache backend for %skey %s", (value ? (int)_oidc_strlen(value) : 0),
			   (cfg->cache.impl->name ? cfg->cache.impl->name : ""), (encrypted ? "encrypted " : ""),
			   (key ? key : ""));
	if (rc == TRUE) {
		oidc_debug(r, "successfully stored %s", msg);
	} else {
		OIDC_METRICS_COUNTER_INC(r, cfg, OM_CACHE_ERROR);
		oidc_warn(r, "could NOT store %s", msg);
	}

	return rc;
}
