/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2002  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <errno.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gnome.h>

#include "support.h"
#include "kiss.h"

KissConfig config;
KissPrefs prefs;
int view;

guchar *palettes[PALETTES * COLS]; /* concrete */

static int numpal, numcol, ckiss, views;
static guchar magic_palette[] =
 { 0, 0, 0,       0, 0, 80,      0, 0, 160,     0, 0, 255,     0, 32, 0,
   0, 32, 80,     0, 32, 160,    0, 32, 255,    0, 71, 0,      0, 71, 80,
   0, 71, 160,    0, 71, 255,    0, 104, 0,     0, 104, 80,    0, 104, 160,
   0, 104, 255,   0, 143, 0,     0, 143, 80,    0, 143, 160,   0, 143, 255,
   0, 176, 0,     0, 176, 80,    0, 176, 160,   0, 176, 255,   0, 215, 0,
   0, 215, 80,    0, 215, 160,   0, 215, 255,   0, 255, 0,     0, 255, 80,
   0, 255, 160,   0, 255, 255,   32, 0, 0,      32, 0, 80,     32, 0, 160,
   32, 0, 255,    32, 32, 0,     32, 32, 80,    32, 32, 160,   32, 32, 255,
   32, 71, 0,     32, 71, 80,    32, 71, 160,   32, 71, 255,   32, 104, 0,
   32, 104, 80,   32, 104, 160,  32, 104, 255,  32, 143, 0,    32, 143, 80,
   32, 143, 160,  32, 143, 255,  32, 176, 0,    32, 176, 80,   32, 176, 160,
   32, 176, 255,  32, 215, 0,    32, 215, 80,   32, 215, 160,  32, 215, 255,
   32, 255, 0,    32, 255, 80,   32, 255, 160,  32, 255, 255,  64, 0, 0,
   64, 0, 80,     64, 0, 160,    64, 0, 255,    64, 32, 0,     64, 32, 80,
   64, 32, 160,   64, 32, 255,   64, 71, 0,     64, 71, 80,    64, 71, 160,  
   64, 71, 255,   64, 104, 0,    64, 104, 80,   64, 104, 160,  64, 104, 255,
   64, 143, 0,    64, 143, 80,   64, 143, 160,  64, 143, 255,  64, 176, 0,
   64, 176, 80,   64, 176, 160,  64, 176, 255,  64, 215, 0,    64, 215, 80,
   64, 215, 160,  64, 215, 255,  64, 255, 0,    64, 255, 80,   64, 255, 160,
   64, 255, 255,  111, 0, 0,     111, 0, 80,    111, 0, 160,   111, 0, 255,
   111, 32, 0,    111, 32, 80,   111, 32, 160,  111, 32, 255,  111, 71, 0,
   111, 71, 80,   111, 71, 160,  111, 71, 255,  111, 104, 0,   111, 104, 80,
   111, 104, 160, 111, 104, 255, 111, 143, 0,   111, 143, 80,  111, 143, 160,
   111, 143, 255, 111, 176, 0,   111, 176, 80,  111, 176, 160, 111, 176, 255,
   111, 215, 0,   111, 215, 80,  111, 215, 160, 111, 215, 255, 111, 255, 0,
   111, 255, 80,  111, 255, 160, 111, 255, 255, 143, 0, 0,     143, 0, 80,   
   143, 0, 160,   143, 0, 255,   143, 32, 0,    143, 32, 80,   143, 32, 160, 
   143, 32, 255,  143, 71, 0,    143, 71, 80,   143, 71, 160,  143, 71, 255,
   143, 104, 0,   143, 104, 80,  143, 104, 160, 143, 104, 255, 143, 143, 0,
   143, 143, 80,  143, 143, 160, 143, 143, 255, 143, 176, 0,   143, 176, 80,
   143, 176, 160, 143, 176, 255, 143, 215, 0,   143, 215, 80,  143, 215, 160,
   143, 215, 255, 143, 255, 0,   143, 255, 80,  143, 255, 160, 143, 255, 255,
   176, 0, 0,     176, 0, 80,    176, 0, 160,   176, 0, 255,   176, 32, 0,
   176, 32, 80,   176, 32, 160,  176, 32, 255,  176, 71, 0,    176, 71, 80,
   176, 71, 160,  176, 71, 255,  176, 104, 0,   176, 104, 80,  176, 104, 160,
   176, 104, 255, 176, 143, 0,   176, 143, 80,  176, 143, 160, 176, 143, 255,
   176, 176, 0,   176, 176, 80,  176, 176, 160, 176, 176, 255, 176, 215, 0,
   176, 215, 80,  176, 215, 160, 176, 215, 255, 176, 255, 0,   176, 255, 80,
   176, 255, 160, 176, 255, 255, 208, 0, 0,     208, 0, 80,    208, 0, 160,
   208, 0, 255,   208, 32, 0,    208, 32, 80,   208, 32, 160,  208, 32, 255,
   208, 71, 0,    208, 71, 80,   208, 71, 160,  208, 71,  255, 208, 104, 0,  
   208, 104, 80,  208, 104, 160, 208, 104, 255, 208, 143, 0,   208, 143, 80, 
   208, 143, 160, 208, 143, 255, 208, 208, 0,   208, 208, 80,  208, 208, 160,
   208, 208, 255, 208, 215, 0,   208, 215, 80,  208, 215, 160, 208, 215, 255,
   208, 255, 0,   208, 255, 80,  208, 255, 160, 208, 255, 255, 255, 0, 0,
   255, 0, 80,    255, 0, 160,   255, 0, 255,   255, 32, 0,    255, 32, 80,
   255, 32, 160,  255, 32, 255,  255, 71, 0,    255, 71, 80,   255, 71, 160, 
   255, 71, 255,  255, 104, 0,   255, 104, 80,  255, 104, 160, 255, 104, 255,
   255, 143, 0,   255, 143, 80,  255, 143, 160, 255, 143, 255, 255, 176, 0,
   255, 176, 80,  255, 176, 160, 255, 176, 255, 255, 215, 0,   255, 215, 80, 
   255, 215, 160, 255, 215, 255, 255, 255, 0,   255, 255, 80,  255, 255, 160,
   255, 255, 255 };

static void fget_line(gchar *s, int size, FILE *stream);
static void parse_size(gchar *entry);
static void parse_bgcolor(gchar *entry);
static void parse_set(gchar *entry);
static void parse_palette(gchar *entry);
static void parse_object(gchar *entry);
static void file_done(void);
static guchar *fake_palette(void);

/* No return value, unlike fgets() */

static void fget_line(gchar *s, int size, FILE *stream) {
  int k, c= '\0';
  static int prev= '\0';

  for(k= 0; k < size - 1; ++k) {
    c= fgetc(stream);
    if (k == 0 && c == '\n' && prev == '\r') {
      c= fgetc(stream); /* next character instead */
    }
    if (c == EOF || c == '\n' || c == '\r') break;
    s[k]= c;
  }

  prev= c;
  s[k]= '\0';
}

void parse_file(const char *filename) {
  FILE *fp;
  gchar *dirname;
  gchar line[1024];
  int k, first_view= 0, script_flag= 0, set_flag= 0;
  gint width, height;
  guint line_no= 0;
  struct stat buf;

  clean_up(); /* clean up anything from previous configs */

  fp= fopen(filename, "r");
  if (fp == NULL) {
    gnome_app_message(GNOME_APP(app), strerror(errno));
    return;
  }

  fstat(fileno(fp), &buf);

  /* move into the appropriate directory */
  dirname= g_dirname(filename);
  chdir(dirname);
  g_free(dirname);

  config.filename= g_strdup(g_basename(filename));

  config.objects= g_hash_table_new(NULL, NULL);
  config.labels= g_hash_table_new(NULL, NULL);
  config.alarms= g_hash_table_new(NULL, NULL);

  /* re-initialise walk-through counters */
  views= 0; numpal= 0; numcol= 0; ckiss= 0;

  view= -1; /* re-initialise set view thingy */

  gnome_appbar_clear_stack(GNOME_APPBAR(appbar));
  gnome_appbar_set_default(GNOME_APPBAR(appbar), _("Reading configuration..."));
  
  for (line_no= 1; !feof(fp); ++line_no) {
    /* note line number for error reporting */
    set_line(line_no);

    if (line_no % 1000 == 0) {
      gnome_appbar_set_progress_percentage(GNOME_APPBAR(appbar),
                                           (ftell(fp) * 1.0) / buf.st_size);
      /* Update GUI */
      while (gtk_events_pending()) gtk_main_iteration();
    }

    fget_line(line, 1024, fp);

    if (strlen(line) > 256) {
      /* just warn, we can parse longer lines */
      log_warning(_("line is too long, should be 256 or less characters"));
    }

    switch (*line) {
    case '%':
      parse_palette(line);
      break;
    case '(':
      parse_size(line);
      break;
    case '[':
      parse_bgcolor(line);
      break;
    case '#':
      if (script_flag) {
        log_warning(_("object definition after @EventHandler"));
      }
      parse_object(line);
      break;
    case ';':
      if (line[1] == '@') {
        if (strstr(line + 2, "EventHandler") != NULL) {
          script_flag= 1;
        } else if (!script_flag) {
          log_warning(_("French KiSS instruction before @EventHandler"));
        } else {
          parse_action(line+2);
        }
      } /* Everything else is a comment */
      break;
    case '$':
      /* following lines may be further set layout */
      set_flag= 1;
      /* fall through */
    case ' ':
      if (set_flag == 1) {
        parse_set(line);
      }
      break;
    default:
      break;
    }
  }
  fclose(fp);

  gnome_appbar_set_progress_percentage(GNOME_APPBAR(appbar), 0.0);
  file_done();

  /* Yes, thanks for asking, these values are arbitrary, if you know a
     way to get better values at run-time please tell me ! */

  width= MIN(gdk_screen_width() - 16, config.width + 32);
  height= MIN(gdk_screen_height() - 16, config.height + 105);

  gtk_window_set_default_size(GTK_WINDOW(app), width, height);
  gtk_drawing_area_size(GTK_DRAWING_AREA(area), config.width, config.height);

  /* Set description */

  config.description= g_strdup_printf("%s%s%s%s", config.filename,
                      (script_flag) ? _(" (FKiSS)") : "",
                      (ckiss) ? _(" (CKiSS)") : "",
                      (numcol > 255) ? _(" (Enhanced)") : "");
                      
  gnome_appbar_set_default(GNOME_APPBAR(appbar), config.description);

  /* Sort out set visibility */

  for (k= SETS - 1; k >= 0; --k) {
    if (views & (1 << k)) {
      gtk_widget_set_sensitive(items[k], 1);
      gtk_widget_set_sensitive(buttons[k], 1);
      first_view= k;
    } else {
      gtk_widget_set_sensitive(items[k], 0);
      gtk_widget_set_sensitive(buttons[k], 0);
    }
  }
  
  /* preliminary events */
  events(config.init);
  events(config.begin);
  events(config.version);

  /* kick start switch_view */
  view= -1;
  switch_view(first_view); /* includes render */
}

static void file_done(void) {
  int k;

  set_line(0); /* done parsing file, no line numbers in errors */

  /* If there were no palettes, add a pseudo-palette. Warnings or
     errors have already been generated if any cells were not CKiSS */

  if (numpal == 0) {
    for (k= 0; k < COLS; ++k) {
      palettes[k]= fake_palette();
    }
  }

  config.rgb_buf = g_new0(guchar, config.row_stride * config.height);
  global_start= config.rgb_buf;
  global_end= config.rgb_buf + (config.row_stride * config.height);
}

/* Note that there is some fairly broken stuf out there, we want to parse
   correct sets perfectly, but doing a good job of incorrect sets is a
   nice-to-have consideration */

/* Hence the spacing is not assumed, if it's wrong that will usually be OK */

static void parse_set(gchar *entry) {
  KissObject *object;
  gchar *token;
  char across[6], down[6];
  static guint objid;
  int x, y, pal= 0;

  if (*entry == '$') {
    if (view >= SETS - 1) {
      log_error(_("too many sets defined, maximum is %u"), SETS);
      return;
    }
    view++;
    if (entry[1] != '0') {
      pal = atoi(entry + 1);
    }
    if (pal < 0 || pal >= COLS) {
      log_error(_("palette set %d out of range"), pal);
      pal = 0;
    }
    config.pal_set[view]= pal;

    entry+= 2; /* skip to position header hopefully */
    objid= 0;
  }

  for (token= entry; *token; token++) {
    if (*token == '*') {
      objid++;
    } else if (sscanf(token, "%5[-0-9],%5[-0-9]", across, down) == 2) {
      x= atoi(across);
      y= atoi(down);
      if ((object= object_id(objid)) != NULL) {
        object->ox[view]= x; object->oy[view]= y; /* set origin position */
        object_set_location(object, x, y);
      } else {
        log_warning(_("object #%u positioned but not defined"), objid);
      }
      token += strlen(across) + strlen(down);
      objid++;
    } else if (!isspace(*token)) {
      log_warning(_("trailing garbage in set layout"));
      break;
    }
  }
}

static void parse_bgcolor(gchar *entry) {
  unsigned int color= 0;

  if (sscanf(entry, "[%u", &color) != 1) {
    log_error(_("invalid syntax in background declaration"));
  } else if (color > 255) {
    log_error(_("invalid color index in background declaration"));
  } else {
    config.bg_color= color;
  }
}

static void parse_size(gchar *entry) {
  int width= 640, height= 480;

  if (sscanf(entry, "(%u,%u)", &width, &height) != 2) {
    log_error(_("invalid syntax in size (width,height) declaration"));
  }

  config.height= height;
  config.width= width;
  config.row_stride= width * 3;
}

static guchar *fake_palette(void) {
  guchar *palette;
  
  palette= g_new0(guchar, 256 * 3);

  if (prefs.default_palette) {
    memcpy(palette, magic_palette, 256 * 3);
  }
  return palette;
}

static void parse_palette(gchar *entry) {
  char filename[32];
  guchar buffer[768];
  guchar *palette= NULL;
  FILE *fp;
  int bits, cols, groups, j, k;

  if (numpal == PALETTES) {
    log_warning(_("too many palettes defined, maximum is %u"), PALETTES);
    return;
  }

  sscanf(entry, "%% %20s ", filename);

  if ((fp= open_any(filename, "r")) == NULL) { /* Fake palette */
    for (j = 0; j < COLS; ++j) {
      palettes[numpal * COLS + j]= fake_palette();
    }
    numcol+= 255;
    numpal++;
    return;
  }

  fread(buffer, 32, 1, fp);

  if (strncmp((char *) buffer, "KiSS", 4)) {
    for (j= 0; j < COLS; j++) {
      palette= fake_palette();
      for (k= 0; k < 16; ++k) {
        palette[k * 3] =     (buffer[k * 2] >> 4)        * 0x11;
        palette[k * 3 + 1] = (buffer[k * 2 + 1] & 0x0f)  * 0x11;
        palette[k * 3 + 2] = (buffer[k * 2]     & 0x0f)  * 0x11;
      }
      palettes[numpal * COLS + j]= palette;
      fread(buffer, 32, 1, fp); /* next 16 colors @ 12-bit */
    }
    numcol+= 15;

  } else { /* New-style palette file */
    bits = buffer[5];
    cols = CLAMP(buffer[8] + 256 * buffer[9], 1, 256);
    groups= CLAMP(buffer[10] + 256 * buffer[11], 1, COLS);

    if (bits == 24) {
      for (j= 0; j < COLS; ++j) {
        palette= fake_palette();
        if (j < groups) {
          fread(palette, cols, 3, fp);
        } else if (j > 0) {
          memcpy(palette, palettes[numpal * COLS], cols * 3);
        }
        palettes[numpal * COLS + j]= palette;
      }
    } else if (bits == 12) {
      for (j= 0; j < COLS; ++j) {
        palette= fake_palette();
        if (j < groups) {
          fread(buffer, cols, 2, fp);
          for (k= 0; k < cols; ++k) {
            palette[k * 3] =     (buffer[k * 2] >> 4)        * 0x11;
            palette[k * 3 + 1] = (buffer[k * 2 + 1] & 0x0f)  * 0x11;
            palette[k * 3 + 2] = (buffer[k * 2]     & 0x0f)  * 0x11;
          }
        } else if (j > 0) {
          memcpy(palette, palettes[numpal * COLS], cols * 3);
        }
        palettes[numpal * COLS + j]= palette;
      }
    } else {
      log_error(_("palette \"%s\" has an unrecognised format"), filename);
      for (j = 0; j < COLS; ++j) {
        palettes[numpal * COLS + j]= fake_palette();
      }
    }
    numcol+= cols - 1;
  }

  numpal++; /* in any case move on */
  fclose(fp);
}

static void parse_object(gchar *entry) {
  unsigned int pal, vis, objid= 0, fix= 0, trans= 0;
  char pair[16], filename[32], *next;
  int digit;
  KissCell *cell;

  /* valid characters for a cell filename defined below */
  if (sscanf(entry, "#%15[.0-9] %31[A-Za-z0-9~_.!-]", pair, filename) != 2) {
    log_error(_("incorrect object syntax"));
    return;
  }

  sscanf(pair, "%u.%u", &objid, &fix);
  if (fix > 32767) {
    log_warning(_("fix exceeds maximum of 32767"));
    fix= 32767;
  }

  next= strchr(entry, ';');
  if (next != NULL) {
    *next= '\0'; /* Separate comments */

    next= strchr(next + 1, '%');
    if (next != NULL && (*(next+1) == 't' || *(next+1) == 'T'))
      trans= strtoul(next + 2, NULL, 10);
  }

  next= strchr(entry, '*');
  if (next != NULL) {
    pal= strtoul(next + 1, NULL, 10);
  } else {
    pal= 0;
  }

  cell= cell_new(objid, pal, filename);
  if (cell == NULL) return; /* error already reported, abort */

  if (cell->argb) {
    ckiss= 1;
  }

  if (cell->index && pal >= numpal) {
    log_error(_("cell uses non-existent palette %u"), pal);
    cell->palette= 0;
  }

  next= strchr(entry, ':');
  if (next != NULL) {
    vis = 0; /* config will specify (even none) */
    do {
      digit= *(next++) - '0';
      if (digit >= 0 && digit < SETS) {
        vis |= (1 << digit);
      }
    } while (*next != '\0');
    if (vis == 0) {
      if (prefs.ignore_colon) {
        vis= (1 << SETS) - 1;
      }
      log_warning(_("cell \"%s\" visiblity is ambiguous"), filename);
    }
  } else {
    /* by default cells are visible in all sets */
    vis = (1 << SETS) - 1;
  }

  if (fix > cell->object->fix) {
    cell->object->fix= fix;
  }
  cell->visible= vis;
  cell->alpha = 255 - trans;
  views |= cell->visible;

  return;
}
