// Copyright 2023 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

package v2_test

import (
	"errors"
	"net/http"
	"testing"

	"github.com/specterops/bloodhound/cmd/api/src/api"
	v2 "github.com/specterops/bloodhound/cmd/api/src/api/v2"
	"github.com/specterops/bloodhound/cmd/api/src/api/v2/apitest"
	"github.com/specterops/bloodhound/cmd/api/src/database/mocks"
	"github.com/specterops/bloodhound/cmd/api/src/model"
	"github.com/specterops/bloodhound/cmd/api/src/model/appcfg"
	"github.com/specterops/bloodhound/cmd/api/src/queries"
	mocks_graph "github.com/specterops/bloodhound/cmd/api/src/queries/mocks"
	"github.com/specterops/bloodhound/packages/go/graphschema/ad"
	"github.com/specterops/bloodhound/packages/go/graphschema/common"
	"github.com/specterops/dawgs/graph"
	"go.uber.org/mock/gomock"
)

func TestResources_GetPathfindingResult(t *testing.T) {
	var (
		mockCtrl  = gomock.NewController(t)
		mockGraph = mocks_graph.NewMockGraph(mockCtrl)
		resources = v2.Resources{GraphQuery: mockGraph}
	)
	defer mockCtrl.Finish()

	apitest.NewHarness(t, resources.GetPathfindingResult).
		Run([]apitest.Case{
			{
				Name: "MissingStartNodeIDParam",
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.BodyContains(output, "Missing query parameter: start_node")
				},
			},
			{
				Name: "MissingEndNodeIDParam",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.BodyContains(output, "Missing query parameter: end_node")
				},
			},
			{
				Name: "GraphDBGetShortestPathsError",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(nil, errors.New("graph error"))
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusInternalServerError)
					apitest.BodyContains(output, "Error:")
				},
			},
			{
				Name: "Success",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(), nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
		})
}

func TestResources_GetShortestPath(t *testing.T) {
	var (
		mockCtrl  = gomock.NewController(t)
		mockGraph = mocks_graph.NewMockGraph(mockCtrl)
		mockDB    = mocks.NewMockDatabase(mockCtrl)
		resources = v2.Resources{GraphQuery: mockGraph, DB: mockDB}
		user      = setupUser()
		userCtx   = setupUserCtx(user)
	)
	defer mockCtrl.Finish()

	apitest.NewHarness(t, resources.GetShortestPath).
		Run([]apitest.Case{
			{
				Name: "MissingStartNodeIDParam",
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
					apitest.BodyContains(output, "Missing query parameter: start_node")
				},
			},
			{
				Name: "MissingEndNodeIDParam",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
					apitest.BodyContains(output, "Missing query parameter: end_node")
				},
			},
			{
				Name: "InvalidRelationshipKindsQuery",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "wrx")
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
					apitest.BodyContains(output, "invalid query parameter 'relationship_kinds': acceptable values should match the format: in|nin:Kind1,Kind2")
				},
			},
			{
				Name: "InvalidRelationshipKindsOperator",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "abcd:Owns,GenericAll,GenericWrite")
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
					apitest.BodyContains(output, "invalid query parameter 'relationship_kinds': acceptable values should match the format: in|nin:Kind1,Kind2")
				},
			},
			{
				Name: "EmptyRelationshipKindsValues",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "abcd:")
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
				},
			},
			{
				Name: "InvalidRelationshipKindsParam",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "in:Owns,avbcs,GenericAll")
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
				},
			},
			{
				Name: "GraphDBGetShortestPathsError",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(nil, errors.New("graph error"))
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusInternalServerError)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
					apitest.BodyContains(output, "graph error")
				},
			},
			{
				Name: "Empty Result Set",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(user)
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(), nil)
					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: true}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusNotFound)
					apitest.UnmarshalBody(output, &api.ErrorWrapper{})
					apitest.BodyContains(output, "Path not found")
				},
			},
			{
				Name: "NotFoundNin",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(user)
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "nin:Owns,GenericAll,AZMGServicePrincipalEndpoint_ReadWrite_All")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(), nil)
					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: true}, nil)

				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusNotFound)
				},
			},
			{
				Name: "NotFoundIn",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(user)
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "in:Owns,GenericAll,GenericWrite,AZMGServicePrincipalEndpoint_ReadWrite_All")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(), nil)
					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: true}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusNotFound)
				},
			},
			{
				Name: "SuccessNin",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(user)
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "nin:Owns,GenericAll,AZMGServicePrincipalEndpoint_ReadWrite_All")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(graph.Path{
							Nodes: []*graph.Node{
								{
									ID:         0,
									Kinds:      graph.Kinds{ad.Entity, ad.Computer},
									Properties: graph.NewProperties(),
								},
								{
									ID:         1,
									Kinds:      graph.Kinds{ad.Entity, ad.User},
									Properties: graph.NewProperties(),
								},
							},
							Edges: []*graph.Relationship{
								{
									ID:         0,
									StartID:    0,
									EndID:      1,
									Kind:       ad.GenericWrite,
									Properties: graph.NewProperties(),
								},
							},
						}), nil)

					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: false}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
			{
				Name: "SuccessIn",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(user)
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "in:Owns,GenericAll,GenericWrite,AZMGServicePrincipalEndpoint_ReadWrite_All")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(graph.Path{
							Nodes: []*graph.Node{
								{
									ID:         0,
									Kinds:      graph.Kinds{ad.Entity, ad.Computer},
									Properties: graph.NewProperties(),
								},
								{
									ID:         1,
									Kinds:      graph.Kinds{ad.Entity, ad.User},
									Properties: graph.NewProperties(),
								},
							},
							Edges: []*graph.Relationship{
								{
									ID:         0,
									StartID:    0,
									EndID:      1,
									Kind:       ad.GenericWrite,
									Properties: graph.NewProperties(),
								},
							},
						}), nil)

					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: false}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
			{
				Name: "NotFoundSingleKind",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(user)
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "someID")
					apitest.AddQueryParam(input, "end_node", "someOtherID")
					apitest.AddQueryParam(input, "relationship_kinds", "in:Owns")
				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(), nil)

					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: true}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusNotFound)
				},
			},
			{
				Name: "FilterETACGraph",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(model.User{
						EnvironmentTargetedAccessControl: []model.EnvironmentTargetedAccessControl{
							{
								UserID:        "",
								EnvironmentID: "12345",
								BigSerial:     model.BigSerial{},
							},
						},
					})
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "0")
					apitest.AddQueryParam(input, "end_node", "1")
					apitest.AddQueryParam(input, "relationship_kinds", "in:GenericWrite")

				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(graph.Path{
							Nodes: []*graph.Node{
								{
									ID:    0,
									Kinds: graph.Kinds{ad.Entity, ad.Computer},
									Properties: graph.AsProperties(graph.PropertyMap{
										ad.DomainSID: "inaccessible",
										common.Name:  "invisible",
									}),
								},
								{
									ID:    1,
									Kinds: graph.Kinds{ad.Entity, ad.User},
									Properties: graph.AsProperties(graph.PropertyMap{
										ad.DomainSID: "12345",
										common.Name:  "visible",
									}),
								},
							},
							Edges: []*graph.Relationship{
								{
									ID:         0,
									StartID:    0,
									EndID:      1,
									Kind:       ad.GenericWrite,
									Properties: graph.NewProperties(),
								},
							},
						}), nil)

					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: true}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.BodyContains(output, "Hidden")
					apitest.BodyContains(output, "visible")
					apitest.BodyNotContains(output, "invisible")
				},
			},
			{
				Name: "Filter ETAC With No Nodes To Filter",
				Input: func(input *apitest.Input) {
					userCtx = setupUserCtx(model.User{
						EnvironmentTargetedAccessControl: []model.EnvironmentTargetedAccessControl{
							{
								UserID:        "",
								EnvironmentID: "12345",
								BigSerial:     model.BigSerial{},
							},
						},
					})
					apitest.SetContext(input, userCtx)

					apitest.AddQueryParam(input, "start_node", "0")
					apitest.AddQueryParam(input, "end_node", "1")
					apitest.AddQueryParam(input, "relationship_kinds", "in:GenericWrite")

				},
				Setup: func() {
					mockGraph.EXPECT().
						GetAllShortestPaths(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
						Return(graph.NewPathSet(graph.Path{
							Nodes: []*graph.Node{
								{
									ID:    0,
									Kinds: graph.Kinds{ad.Entity, ad.Computer},
									Properties: graph.AsProperties(graph.PropertyMap{
										ad.DomainSID: "12345",
										common.Name:  "visible-2",
									}),
								},
								{
									ID:    1,
									Kinds: graph.Kinds{ad.Entity, ad.User},
									Properties: graph.AsProperties(graph.PropertyMap{
										ad.DomainSID: "12345",
										common.Name:  "visible",
									}),
								},
							},
							Edges: []*graph.Relationship{
								{
									ID:         0,
									StartID:    0,
									EndID:      1,
									Kind:       ad.GenericWrite,
									Properties: graph.NewProperties(),
								},
							},
						}), nil)

					mockDB.EXPECT().GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).Return(appcfg.FeatureFlag{Enabled: true}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.BodyNotContains(output, "Hidden")
					apitest.BodyContains(output, "visible")
					apitest.BodyContains(output, "visible-2")
				},
			},
		})
}

func TestResources_GetSearchResult(t *testing.T) {
	var (
		mockCtrl   = gomock.NewController(t)
		mockGraph  = mocks_graph.NewMockGraph(mockCtrl)
		mockDB     = mocks.NewMockDatabase(mockCtrl)
		resources  = v2.Resources{GraphQuery: mockGraph, DB: mockDB}
		user       = setupUser()
		userCtx    = setupUserCtx(user)
		allEnvUser = model.User{
			AllEnvironments: true,
		}
		allEnvUserCtx = setupUserCtx(allEnvUser)
		etacUser      = model.User{
			AllEnvironments: false,
			EnvironmentTargetedAccessControl: []model.EnvironmentTargetedAccessControl{
				{EnvironmentID: "testenv"},
			},
		}
		etacUserCtx = setupUserCtx(etacUser)
	)
	defer mockCtrl.Finish()

	apitest.NewHarness(t, resources.GetSearchResult).
		Run([]apitest.Case{
			{
				Name: "MissingSearchParam",
				Input: func(input *apitest.Input) {
					apitest.SetContext(input, userCtx)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.BodyContains(output, "Expected search parameter to be set.")
				},
			},
			{
				Name: "TooManySearchParams",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "query", "some other invalid query")
					apitest.SetContext(input, userCtx)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.BodyContains(output, "Expected only one search value.")
				},
			},
			{
				Name: "TooManySearchTypeParams",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "type", "search type")
					apitest.AddQueryParam(input, "type", "another search type")
					apitest.SetContext(input, userCtx)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.BodyContains(output, "Expected only one search type.")
				},
			},
			{
				Name: "FeatureFlagDatabaseError -- Open Graph",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.SetContext(input, userCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{}, errors.New("database error"))
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusInternalServerError)
					apitest.BodyContains(output, "an internal error has occurred that is preventing the service from servicing this request")
				},
			},
			{
				Name: "GraphDBSearchByNameOrObjectIDError",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.SetContext(input, userCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)
					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), true, "some query", queries.SearchTypeFuzzy).
						Return(nil, errors.New("graph error"))
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusBadRequest)
					apitest.BodyContains(output, "Error getting search results:")
				},
			},
			{
				Name: "DBGetCustomNodeKindsError -- should still return results",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.SetContext(input, userCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)
					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), true, "some query", queries.SearchTypeFuzzy).
						Return(graph.NewNodeSet(), nil)
					mockDB.EXPECT().GetCustomNodeKinds(gomock.Any()).Return([]model.CustomNodeKind{}, errors.New("error"))
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).
						Return(appcfg.FeatureFlag{Enabled: false}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
			{
				Name: "FeatureFlagDatabaseError -- ETAC",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "type", "fuzzy")
					apitest.SetContext(input, userCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: false}, nil)
					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), false, "some query", queries.SearchTypeFuzzy).
						Return(graph.NewNodeSet(), nil)
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).
						Return(appcfg.FeatureFlag{}, errors.New("database error"))
					mockDB.EXPECT().GetCustomNodeKinds(gomock.Any()).Return([]model.CustomNodeKind{}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusInternalServerError)
				},
			},
			{
				Name: "Success -- include OpenGraph results",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "type", "fuzzy")
					apitest.SetContext(input, userCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)
					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), true, "some query", queries.SearchTypeFuzzy).
						Return(graph.NewNodeSet(), nil)
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).
						Return(appcfg.FeatureFlag{Enabled: false}, nil)
					mockDB.EXPECT().GetCustomNodeKinds(gomock.Any()).Return([]model.CustomNodeKind{}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
			{
				Name: "Success -- exclude OpenGraph results",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "type", "fuzzy")
					apitest.SetContext(input, userCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: false}, nil)
					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), false, "some query", queries.SearchTypeFuzzy).
						Return(graph.NewNodeSet(), nil)
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).
						Return(appcfg.FeatureFlag{Enabled: false}, nil)
					mockDB.EXPECT().GetCustomNodeKinds(gomock.Any()).Return([]model.CustomNodeKind{}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
			{
				Name: "Success -- ETAC enabled,user has all environments",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "type", "fuzzy")
					apitest.SetContext(input, allEnvUserCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)

					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), true, "some query", queries.SearchTypeFuzzy).
						Return(graph.NewNodeSet(), nil)

					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)
					mockDB.EXPECT().GetCustomNodeKinds(gomock.Any()).Return([]model.CustomNodeKind{}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
				},
			},
			{
				Name: "Success -- ETAC enabled,user has limited access",
				Input: func(input *apitest.Input) {
					apitest.AddQueryParam(input, "query", "some query")
					apitest.AddQueryParam(input, "type", "fuzzy")
					apitest.SetContext(input, etacUserCtx)
				},
				Setup: func() {
					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureOpenGraphSearch).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)

					nodeSet := graph.NewNodeSet()

					accessibleNode := &graph.Node{
						ID: 1,
						Properties: &graph.Properties{
							Map: map[string]any{
								"domainsid": "testenv"},
						},
					}
					hiddenNode := &graph.Node{
						ID:    2,
						Kinds: graph.Kinds{ad.Computer},
						Properties: &graph.Properties{
							Map: map[string]any{
								"domainsid": "restricted",
								"name":      "restricted",
							},
						},
					}
					nodeSet.Add(accessibleNode)
					nodeSet.Add(hiddenNode)
					mockGraph.EXPECT().
						SearchByNameOrObjectID(gomock.Any(), true, "some query", queries.SearchTypeFuzzy).
						Return(nodeSet, nil)

					mockDB.EXPECT().
						GetFlagByKey(gomock.Any(), appcfg.FeatureETAC).
						Return(appcfg.FeatureFlag{Enabled: true}, nil)
					mockDB.EXPECT().GetCustomNodeKinds(gomock.Any()).Return([]model.CustomNodeKind{}, nil)
				},
				Test: func(output apitest.Output) {
					apitest.StatusCode(output, http.StatusOK)
					apitest.BodyContains(output, "testenv")
					apitest.BodyContains(output, "** Hidden Computer Object **")
					apitest.BodyNotContains(output, "restricted")
				},
			},
		})
}
