// Copyright 2025 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

import userEvent from '@testing-library/user-event';
import { rest } from 'msw';
import { setupServer } from 'msw/node';
import { render, screen, within } from '../../test-utils';
import { AppIcon } from '../AppIcon';
import MainNav from './MainNav';
import { MainNavData, MainNavDataListItem, MainNavLogoDataObject } from './types';

const MainNavLogoData: MainNavLogoDataObject = {
    project: {
        route: '/',
        icon: <AppIcon.BHCELogo size={24} />,
    },
    specterOps: {
        image: {
            imageUrl: `/test`,
            dimensions: { height: 40, width: 165 },
            classes: 'ml-4',
            altText: 'BHE Text Logo',
        },
    },
};
const MainNavPrimaryListData: MainNavDataListItem[] = [
    {
        label: 'Link Item',
        icon: <AppIcon.LineChart size={24} />,
        route: '/test',
        testId: 'global_nav-test-link',
    },
    {
        label: 'Link Item 2',
        icon: <AppIcon.LineChart size={24} />,
        route: '/secondroute',
        testId: 'global_nav-test-link-2',
    },
];

const handleClick = vi.fn();

const MainNavSecondaryListData: MainNavDataListItem[] = [
    {
        label: 'Action Item',
        icon: <AppIcon.LineChart size={24} />,
        functionHandler: handleClick,
        testId: 'global_nav-test-action',
    },
];

const mainNavData: MainNavData = {
    logo: MainNavLogoData,
    primaryList: MainNavPrimaryListData,
    secondaryList: MainNavSecondaryListData,
};

const currentVersionNumber = 'v999.999.999';

const server = setupServer(
    rest.get(`/api/version`, async (_req, res, ctx) => {
        return res(
            ctx.json({
                data: {
                    API: {
                        current_version: 'v2',
                        deprecated_version: 'v1',
                    },
                    server_version: currentVersionNumber,
                },
            })
        );
    })
);
beforeAll(() => server.listen());
afterEach(() => server.resetHandlers());
afterAll(() => server.close());

describe('MainNav', () => {
    const user = userEvent.setup();

    beforeEach(() => {
        render(<MainNav mainNavData={mainNavData} />);
    });
    it('should render a nav element with logo, two lists, a version number and a powered by', () => {
        expect(screen.getByRole('navigation')).toBeInTheDocument();
        expect(screen.getByTestId('global_nav-home')).toBeInTheDocument();
        expect(screen.getByTestId('global_nav-primary-list')).toBeInTheDocument();
        expect(screen.getByTestId('global_nav-secondary-list')).toBeInTheDocument();
        expect(screen.getByTestId('global_nav-version-number')).toBeInTheDocument();
        expect(screen.getByTestId('global_nav-powered-by')).toBeInTheDocument();
    });
    it('should render a navigation list item', async () => {
        const testLinkItem = MainNavPrimaryListData[0];

        const primaryList = await screen.findByTestId('global_nav-primary-list');
        const linkItem = await within(primaryList).getAllByTestId('global_nav-test-link')[0];
        const linkItemIcon = await within(primaryList).getAllByTestId('global_nav-item-label-icon')[0];
        const linkItemText = await within(primaryList).findByText(testLinkItem.label as string);

        expect(linkItem).toBeInTheDocument();
        expect(linkItem).toHaveAttribute('href', testLinkItem.route);
        expect(linkItemIcon).toBeInTheDocument();
        expect(linkItemText).toBeInTheDocument();
    });
    it('should render action list item that handles a function', async () => {
        const testLinkItem = MainNavSecondaryListData[0];

        const secondaryList = await screen.findByTestId('global_nav-secondary-list');
        const actionItem = await within(secondaryList).findByRole('button');
        const actionItemIcon = await within(secondaryList).findByTestId('global_nav-item-label-icon');
        const actionItemText = await within(secondaryList).findByText(testLinkItem.label as string);

        expect(actionItem).toBeInTheDocument();
        expect(actionItemIcon).toBeInTheDocument();
        expect(actionItemText).toBeInTheDocument();

        await user.click(actionItem);

        expect(testLinkItem.functionHandler).toBeCalled();
    });
    it('should render a label and version number when expanded', async () => {
        const MainNavBar = await screen.findByRole('navigation');
        expect(MainNavBar).toHaveClass('group');

        const versionNumberContainer = await within(MainNavBar).findByTestId('global_nav-version-number');
        const versionNumberLabel = await within(versionNumberContainer).findByText(
            `BloodHound: ${currentVersionNumber}`
        );

        // ---- collapsed classes ----
        expect(versionNumberLabel).toHaveClass('hidden');
        expect(versionNumberLabel).toHaveClass('opacity-0');
        // ---- collapsed classes ----

        // ---- classes displayed on hover ----
        expect(versionNumberLabel).toHaveClass('group-hover:opacity-100');
        expect(versionNumberLabel).toHaveClass('group-hover:block');
        // ---- classes displayed on hover ----
    });
    it('has styles that hide icon labels when the nav is collapsed and show labels when its expanded', async () => {
        // This test is a quite naive but its purpose is to essentially create a contract between the TSX and styles.
        // Ideally we could check for these styles and then check for labels outside the bounding rect and icons within it.
        // ... but jsdom would rather we not do that
        const MainNavBar = screen.getByRole('navigation');

        expect(MainNavBar).toHaveClass('w-nav-width hover:w-nav-width-expanded hover:overflow-x-hidden');

        // MainNavLogo
        const navLogo = within(MainNavBar).getByTestId('global_nav-home');
        expect(navLogo).toHaveClass('overflow-hidden');

        // MainNavListItem
        const navItems = within(MainNavBar).getAllByRole('listitem');
        navItems.every((item) => expect(item).toHaveClass('overflow-hidden'));
    });
    it('should style the powered-by to display when nav is expanded', async () => {
        const MainNavBar = screen.getByRole('navigation');
        expect(MainNavBar).toHaveClass('group');

        const poweredByTextContainer = await within(MainNavBar).findByTestId('global_nav-powered-by');
        const poweredByText = await within(poweredByTextContainer).findByText(/powered by/i);
        expect(poweredByText).toBeInTheDocument();

        // ---- collapsed classes ----
        expect(poweredByText).toHaveClass('hidden');
        expect(poweredByText).toHaveClass('opacity-0');
        // ---- collapsed classes ----

        // ---- classes displayed on hover ----
        expect(poweredByText).toHaveClass('group-hover:opacity-100');
        expect(poweredByText).toHaveClass('group-hover:flex');
        // ---- classes displayed on hover ----
    });
    it('should have a .z-nav class', () => {
        const navbarElement = screen.getByRole('navigation');
        expect(navbarElement).toHaveClass('z-nav');
    });
});

describe('Main Nav Route Highlighting', () => {
    it('should highlight selected route', () => {
        render(<MainNav mainNavData={mainNavData} />, {
            route: '/test',
        });
        expect(window.location.pathname).toBe('/test');
        const elem = screen.getByTestId('global_nav-test-link').closest('li');
        expect(elem).toHaveClass('bg-neutral-light-4');
    });
    it('should highlight main nav route when navigating to child route', () => {
        render(<MainNav mainNavData={mainNavData} />, {
            route: '/secondroute/child',
        });
        const selected = screen.getByTestId('global_nav-test-link-2').closest('li');
        const unselected = screen.getByTestId('global_nav-test-link').closest('li');
        expect(selected).toHaveClass('bg-neutral-light-4');
        expect(unselected).not.toHaveClass('bg-neutral-light-4');
    });
});
