/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  AbsoluteTime,
  HttpStatusCode,
  TalerError,
  TalerErrorCode,
  TalerMerchantApi,
  assertUnreachable
} from "@gnu-taler/taler-util";
import {
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { JumpToElementById } from "../../../../components/form/JumpToElementById.js";
import { useSessionContext } from "../../../../context/session.js";
import {
  InstanceOrderFilter,
  useInstanceOrders,
  useOrderDetails,
} from "../../../../hooks/order.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { ListPage } from "./ListPage.js";
import { RefundModal } from "./Table.js";


const TALER_SCREEN_ID = 46;

interface Props {
  onSelect: (id: string) => void;
  onCreate: () => void;
}

export default function OrderList({ onCreate, onSelect }: Props): VNode {
  const [filter, setFilter] = useState<InstanceOrderFilter>({ paid: false });
  const [orderToBeRefunded, setOrderToBeRefunded] = useState<
    TalerMerchantApi.OrderHistoryEntry | undefined
  >(undefined);

  const setNewDate = (date?: AbsoluteTime): void =>
    setFilter((prev) => ({ ...prev, date }));

  const result = useInstanceOrders(filter, (d) =>
    setFilter({ ...filter, position: d }),
  );
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const { state: session, lib } = useSessionContext();

  const { i18n } = useTranslationContext();

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  const isNotPaidActive = filter.paid === false ? "is-active" : "";
  const isPaidActive =
    filter.paid === true && filter.wired === undefined ? "is-active" : "";
  const isRefundedActive = filter.refunded === true ? "is-active" : "";
  const isNotWiredActive =
    filter.wired === false && filter.paid === true ? "is-active" : "";
  const isWiredActive = filter.wired === true ? "is-active" : "";
  const isAllActive =
    filter.paid === undefined &&
      filter.refunded === undefined &&
      filter.wired === undefined
      ? "is-active"
      : "";

  const data = {} as TalerMerchantApi.RefundRequest
  const refund = safeFunctionHandler(
    lib.instance.addRefund.bind(lib.instance),
    !session.token || !orderToBeRefunded ? undefined : [session.token, orderToBeRefunded.order_id, data],
  );
  refund.onSuccess = () => {
    setOrderToBeRefunded(undefined)
  };
  refund.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.Forbidden:
        return i18n.str`Forbidden.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Conflict.`;
      case HttpStatusCode.Gone:
        return i18n.str`Gone.`;
      case HttpStatusCode.UnavailableForLegalReasons:
        return i18n.str`UnavailableForLegalReasons.`;
    }
  };
  return (
    <section class="section is-main-section">
      <LocalNotificationBannerBulma notification={notification} />

      <JumpToElementById
        onSelect={onSelect}
        description={i18n.str`Jump to order with the given product ID`}
        placeholder={i18n.str`Order id`}
      />

      <ListPage
        orders={result.body.map((o) => ({ ...o, id: o.order_id }))}
        onLoadMoreBefore={result.loadFirst}
        onLoadMoreAfter={result.loadNext}
        onSelectOrder={(order) => onSelect(order.id)}
        onRefundOrder={(value) => setOrderToBeRefunded(value)}
        isAllActive={isAllActive}
        isNotWiredActive={isNotWiredActive}
        isWiredActive={isWiredActive}
        isPaidActive={isPaidActive}
        isNotPaidActive={isNotPaidActive}
        isRefundedActive={isRefundedActive}
        jumpToDate={filter.date}
        onSelectDate={setNewDate}
        onCreate={onCreate}
        onShowAll={() => setFilter({})}
        onShowNotPaid={() => setFilter({ paid: false })}
        onShowPaid={() => setFilter({ paid: true })}
        onShowRefunded={() => setFilter({ refunded: true })}
        onShowNotWired={() => setFilter({ wired: false, paid: true })}
        onShowWired={() => setFilter({ wired: true })}
      />

      {orderToBeRefunded && (
        <RefundModalForTable
          id={orderToBeRefunded.order_id}
          onCancel={() => setOrderToBeRefunded(undefined)}
          onConfirmed={() => setOrderToBeRefunded(undefined)}
        />
      )}
    </section>
  );
}

interface RefundProps {
  id: string;
  onCancel: () => void;
  onConfirmed: () => void;
}

function RefundModalForTable({ id, onConfirmed, onCancel }: RefundProps): VNode {
  const { i18n } = useTranslationContext();
  const result = useOrderDetails(id);

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case TalerErrorCode.MERCHANT_GENERIC_INSTANCE_UNKNOWN: {
        return <i18n.Translate>Instance unknown</i18n.Translate>;
      }
      case TalerErrorCode.MERCHANT_GENERIC_ORDER_UNKNOWN: {
        return <i18n.Translate>Order unknown</i18n.Translate>;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  return (
    <RefundModal
      id={id}
      order={result.body}
      onCancel={onCancel}
      onConfirmed={onConfirmed}
    />
  );
}

async function copyToClipboard(text: string): Promise<void> {
  console.log("copied", text);
  return navigator.clipboard.writeText(text);
}
