// SPDX-License-Identifier: MIT
/*
 * Copyright 2022 Advanced Micro Devices, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors: AMD
 *
 */
#include "dcn32_fpu.h"
#include "dcn32/dcn32_resource.h"
#include "dcn20/dcn20_resource.h"
#include "display_mode_vba_util_32.h"
#include "dml/dcn32/display_mode_vba_32.h"
// We need this includes for WATERMARKS_* defines
#include "clk_mgr/dcn32/dcn32_smu13_driver_if.h"
#include "dcn30/dcn30_resource.h"
#include "link.h"
#include "dc_state_priv.h"

#define DC_LOGGER_INIT(logger)

static const struct subvp_high_refresh_list subvp_high_refresh_list = {
			.min_refresh = 120,
			.max_refresh = 175,
			.res = {
				{.width = 3840, .height = 2160, },
				{.width = 3440, .height = 1440, },
				{.width = 2560, .height = 1440, },
				{.width = 1920, .height = 1080, }},
};

static const struct subvp_active_margin_list subvp_active_margin_list = {
			.min_refresh = 55,
			.max_refresh = 65,
			.res = {
				{.width = 2560, .height = 1440, },
				{.width = 1920, .height = 1080, }},
};

struct _vcs_dpi_ip_params_st dcn3_2_ip = {
	.gpuvm_enable = 0,
	.gpuvm_max_page_table_levels = 4,
	.hostvm_enable = 0,
	.rob_buffer_size_kbytes = 128,
	.det_buffer_size_kbytes = DCN3_2_DEFAULT_DET_SIZE,
	.config_return_buffer_size_in_kbytes = 1280,
	.compressed_buffer_segment_size_in_kbytes = 64,
	.meta_fifo_size_in_kentries = 22,
	.zero_size_buffer_entries = 512,
	.compbuf_reserved_space_64b = 256,
	.compbuf_reserved_space_zs = 64,
	.dpp_output_buffer_pixels = 2560,
	.opp_output_buffer_lines = 1,
	.pixel_chunk_size_kbytes = 8,
	.alpha_pixel_chunk_size_kbytes = 4,
	.min_pixel_chunk_size_bytes = 1024,
	.dcc_meta_buffer_size_bytes = 6272,
	.meta_chunk_size_kbytes = 2,
	.min_meta_chunk_size_bytes = 256,
	.writeback_chunk_size_kbytes = 8,
	.ptoi_supported = false,
	.num_dsc = 4,
	.maximum_dsc_bits_per_component = 12,
	.maximum_pixels_per_line_per_dsc_unit = 6016,
	.dsc422_native_support = true,
	.is_line_buffer_bpp_fixed = true,
	.line_buffer_fixed_bpp = 57,
	.line_buffer_size_bits = 1171920,
	.max_line_buffer_lines = 32,
	.writeback_interface_buffer_size_kbytes = 90,
	.max_num_dpp = 4,
	.max_num_otg = 4,
	.max_num_hdmi_frl_outputs = 1,
	.max_num_wb = 1,
	.max_dchub_pscl_bw_pix_per_clk = 4,
	.max_pscl_lb_bw_pix_per_clk = 2,
	.max_lb_vscl_bw_pix_per_clk = 4,
	.max_vscl_hscl_bw_pix_per_clk = 4,
	.max_hscl_ratio = 6,
	.max_vscl_ratio = 6,
	.max_hscl_taps = 8,
	.max_vscl_taps = 8,
	.dpte_buffer_size_in_pte_reqs_luma = 64,
	.dpte_buffer_size_in_pte_reqs_chroma = 34,
	.dispclk_ramp_margin_percent = 1,
	.max_inter_dcn_tile_repeaters = 8,
	.cursor_buffer_size = 16,
	.cursor_chunk_size = 2,
	.writeback_line_buffer_buffer_size = 0,
	.writeback_min_hscl_ratio = 1,
	.writeback_min_vscl_ratio = 1,
	.writeback_max_hscl_ratio = 1,
	.writeback_max_vscl_ratio = 1,
	.writeback_max_hscl_taps = 1,
	.writeback_max_vscl_taps = 1,
	.dppclk_delay_subtotal = 47,
	.dppclk_delay_scl = 50,
	.dppclk_delay_scl_lb_only = 16,
	.dppclk_delay_cnvc_formatter = 28,
	.dppclk_delay_cnvc_cursor = 6,
	.dispclk_delay_subtotal = 125,
	.dynamic_metadata_vm_enabled = false,
	.odm_combine_4to1_supported = false,
	.dcc_supported = true,
	.max_num_dp2p0_outputs = 2,
	.max_num_dp2p0_streams = 4,
};

struct _vcs_dpi_soc_bounding_box_st dcn3_2_soc = {
	.clock_limits = {
		{
			.state = 0,
			.dcfclk_mhz = 1564.0,
			.fabricclk_mhz = 2500.0,
			.dispclk_mhz = 2150.0,
			.dppclk_mhz = 2150.0,
			.phyclk_mhz = 810.0,
			.phyclk_d18_mhz = 667.0,
			.phyclk_d32_mhz = 625.0,
			.socclk_mhz = 1200.0,
			.dscclk_mhz = 716.667,
			.dram_speed_mts = 18000.0,
			.dtbclk_mhz = 1564.0,
		},
	},
	.num_states = 1,
	.sr_exit_time_us = 42.97,
	.sr_enter_plus_exit_time_us = 49.94,
	.sr_exit_z8_time_us = 285.0,
	.sr_enter_plus_exit_z8_time_us = 320,
	.writeback_latency_us = 12.0,
	.round_trip_ping_latency_dcfclk_cycles = 263,
	.urgent_latency_pixel_data_only_us = 4.0,
	.urgent_latency_pixel_mixed_with_vm_data_us = 4.0,
	.urgent_latency_vm_data_only_us = 4.0,
	.fclk_change_latency_us = 25,
	.usr_retraining_latency_us = 2,
	.smn_latency_us = 2,
	.mall_allocated_for_dcn_mbytes = 64,
	.urgent_out_of_order_return_per_channel_pixel_only_bytes = 4096,
	.urgent_out_of_order_return_per_channel_pixel_and_vm_bytes = 4096,
	.urgent_out_of_order_return_per_channel_vm_only_bytes = 4096,
	.pct_ideal_sdp_bw_after_urgent = 90.0,
	.pct_ideal_fabric_bw_after_urgent = 67.0,
	.pct_ideal_dram_sdp_bw_after_urgent_pixel_only = 20.0,
	.pct_ideal_dram_sdp_bw_after_urgent_pixel_and_vm = 60.0,
	.pct_ideal_dram_sdp_bw_after_urgent_vm_only = 30.0,
	.pct_ideal_dram_bw_after_urgent_strobe = 67.0,
	.max_avg_sdp_bw_use_normal_percent = 80.0,
	.max_avg_fabric_bw_use_normal_percent = 60.0,
	.max_avg_dram_bw_use_normal_strobe_percent = 50.0,
	.max_avg_dram_bw_use_normal_percent = 15.0,
	.num_chans = 24,
	.dram_channel_width_bytes = 2,
	.fabric_datapath_to_dcn_data_return_bytes = 64,
	.return_bus_width_bytes = 64,
	.downspread_percent = 0.38,
	.dcn_downspread_percent = 0.5,
	.dram_clock_change_latency_us = 400,
	.dispclk_dppclk_vco_speed_mhz = 4300.0,
	.do_urgent_latency_adjustment = true,
	.urgent_latency_adjustment_fabric_clock_component_us = 1.0,
	.urgent_latency_adjustment_fabric_clock_reference_mhz = 3000,
};

static bool dcn32_apply_merge_split_flags_helper(struct dc *dc, struct dc_state *context,
	bool *repopulate_pipes, int *split, bool *merge);

void dcn32_build_wm_range_table_fpu(struct clk_mgr_internal *clk_mgr)
{
	/* defaults */
	double pstate_latency_us = clk_mgr->base.ctx->dc->dml.soc.dram_clock_change_latency_us;
	double fclk_change_latency_us = clk_mgr->base.ctx->dc->dml.soc.fclk_change_latency_us;
	double sr_exit_time_us = clk_mgr->base.ctx->dc->dml.soc.sr_exit_time_us;
	double sr_enter_plus_exit_time_us = clk_mgr->base.ctx->dc->dml.soc.sr_enter_plus_exit_time_us;
	/* For min clocks use as reported by PM FW and report those as min */
	uint16_t min_uclk_mhz			= clk_mgr->base.bw_params->clk_table.entries[0].memclk_mhz;
	uint16_t min_dcfclk_mhz			= clk_mgr->base.bw_params->clk_table.entries[0].dcfclk_mhz;
	uint16_t setb_min_uclk_mhz		= min_uclk_mhz;
	uint16_t dcfclk_mhz_for_the_second_state = clk_mgr->base.ctx->dc->dml.soc.clock_limits[2].dcfclk_mhz;

	dc_assert_fp_enabled();

	/* For Set B ranges use min clocks state 2 when available, and report those to PM FW */
	if (dcfclk_mhz_for_the_second_state)
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].pmfw_breakdown.min_dcfclk = dcfclk_mhz_for_the_second_state;
	else
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].pmfw_breakdown.min_dcfclk = clk_mgr->base.bw_params->clk_table.entries[0].dcfclk_mhz;

	if (clk_mgr->base.bw_params->clk_table.entries[2].memclk_mhz)
		setb_min_uclk_mhz = clk_mgr->base.bw_params->clk_table.entries[2].memclk_mhz;

	/* Set A - Normal - default values */
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].valid = true;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].dml_input.pstate_latency_us = pstate_latency_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].dml_input.fclk_change_latency_us = fclk_change_latency_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].dml_input.sr_exit_time_us = sr_exit_time_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].dml_input.sr_enter_plus_exit_time_us = sr_enter_plus_exit_time_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].pmfw_breakdown.wm_type = WATERMARKS_CLOCK_RANGE;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].pmfw_breakdown.min_dcfclk = min_dcfclk_mhz;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].pmfw_breakdown.max_dcfclk = 0xFFFF;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].pmfw_breakdown.min_uclk = min_uclk_mhz;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_A].pmfw_breakdown.max_uclk = 0xFFFF;

	/* Set B - Performance - higher clocks, using DPM[2] DCFCLK and UCLK */
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].valid = true;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].dml_input.pstate_latency_us = pstate_latency_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].dml_input.fclk_change_latency_us = fclk_change_latency_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].dml_input.sr_exit_time_us = sr_exit_time_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].dml_input.sr_enter_plus_exit_time_us = sr_enter_plus_exit_time_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].pmfw_breakdown.wm_type = WATERMARKS_CLOCK_RANGE;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].pmfw_breakdown.max_dcfclk = 0xFFFF;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].pmfw_breakdown.min_uclk = setb_min_uclk_mhz;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_B].pmfw_breakdown.max_uclk = 0xFFFF;

	/* Set C - Dummy P-State - P-State latency set to "dummy p-state" value */
	/* 'DalDummyClockChangeLatencyNs' registry key option set to 0x7FFFFFFF can be used to disable Set C for dummy p-state */
	if (clk_mgr->base.ctx->dc->bb_overrides.dummy_clock_change_latency_ns != 0x7FFFFFFF) {
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].valid = true;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].dml_input.pstate_latency_us = 50;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].dml_input.fclk_change_latency_us = fclk_change_latency_us;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].dml_input.sr_exit_time_us = sr_exit_time_us;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].dml_input.sr_enter_plus_exit_time_us = sr_enter_plus_exit_time_us;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].pmfw_breakdown.wm_type = WATERMARKS_DUMMY_PSTATE;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].pmfw_breakdown.min_dcfclk = min_dcfclk_mhz;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].pmfw_breakdown.max_dcfclk = 0xFFFF;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].pmfw_breakdown.min_uclk = min_uclk_mhz;
		clk_mgr->base.bw_params->wm_table.nv_entries[WM_C].pmfw_breakdown.max_uclk = 0xFFFF;
		clk_mgr->base.bw_params->dummy_pstate_table[0].dram_speed_mts = clk_mgr->base.bw_params->clk_table.entries[0].memclk_mhz * 16;
		clk_mgr->base.bw_params->dummy_pstate_table[0].dummy_pstate_latency_us = 50;
		clk_mgr->base.bw_params->dummy_pstate_table[1].dram_speed_mts = clk_mgr->base.bw_params->clk_table.entries[1].memclk_mhz * 16;
		clk_mgr->base.bw_params->dummy_pstate_table[1].dummy_pstate_latency_us = 9;
		clk_mgr->base.bw_params->dummy_pstate_table[2].dram_speed_mts = clk_mgr->base.bw_params->clk_table.entries[2].memclk_mhz * 16;
		clk_mgr->base.bw_params->dummy_pstate_table[2].dummy_pstate_latency_us = 8;
		clk_mgr->base.bw_params->dummy_pstate_table[3].dram_speed_mts = clk_mgr->base.bw_params->clk_table.entries[3].memclk_mhz * 16;
		clk_mgr->base.bw_params->dummy_pstate_table[3].dummy_pstate_latency_us = 5;
	}
	/* Set D - MALL - SR enter and exit time specific to MALL, TBD after bringup or later phase for now use DRAM values / 2 */
	/* For MALL DRAM clock change latency is N/A, for watermak calculations use lowest value dummy P state latency */
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].valid = true;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].dml_input.pstate_latency_us = clk_mgr->base.bw_params->dummy_pstate_table[3].dummy_pstate_latency_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].dml_input.fclk_change_latency_us = fclk_change_latency_us;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].dml_input.sr_exit_time_us = sr_exit_time_us / 2; // TBD
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].dml_input.sr_enter_plus_exit_time_us = sr_enter_plus_exit_time_us / 2; // TBD
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].pmfw_breakdown.wm_type = WATERMARKS_MALL;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].pmfw_breakdown.min_dcfclk = min_dcfclk_mhz;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].pmfw_breakdown.max_dcfclk = 0xFFFF;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].pmfw_breakdown.min_uclk = min_uclk_mhz;
	clk_mgr->base.bw_params->wm_table.nv_entries[WM_D].pmfw_breakdown.max_uclk = 0xFFFF;
}

/*
 * Finds dummy_latency_index when MCLK switching using firmware based
 * vblank stretch is enabled. This function will iterate through the
 * table of dummy pstate latencies until the lowest value that allows
 * dm_allow_self_refresh_and_mclk_switch to happen is found
 */
int dcn32_find_dummy_latency_index_for_fw_based_mclk_switch(struct dc *dc,
							    struct dc_state *context,
							    display_e2e_pipe_params_st *pipes,
							    int pipe_cnt,
							    int vlevel)
{
	const int max_latency_table_entries = 4;
	struct vba_vars_st *vba = &context->bw_ctx.dml.vba;
	int dummy_latency_index = 0;
	enum clock_change_support temp_clock_change_support = vba->DRAMClockChangeSupport[vlevel][context->bw_ctx.dml.vba.maxMpcComb];

	dc_assert_fp_enabled();

	while (dummy_latency_index < max_latency_table_entries) {
		if (temp_clock_change_support != dm_dram_clock_change_unsupported)
			vba->DRAMClockChangeSupport[vlevel][context->bw_ctx.dml.vba.maxMpcComb] = temp_clock_change_support;
		context->bw_ctx.dml.soc.dram_clock_change_latency_us =
				dc->clk_mgr->bw_params->dummy_pstate_table[dummy_latency_index].dummy_pstate_latency_us;
		dcn32_internal_validate_bw(dc, context, pipes, &pipe_cnt, &vlevel, false);

		/* for subvp + DRR case, if subvp pipes are still present we support pstate */
		if (vba->DRAMClockChangeSupport[vlevel][vba->maxMpcComb] == dm_dram_clock_change_unsupported &&
				dcn32_subvp_in_use(dc, context))
			vba->DRAMClockChangeSupport[vlevel][context->bw_ctx.dml.vba.maxMpcComb] = temp_clock_change_support;

		if (vlevel < context->bw_ctx.dml.vba.soc.num_states &&
				vba->DRAMClockChangeSupport[vlevel][vba->maxMpcComb] != dm_dram_clock_change_unsupported)
			break;

		dummy_latency_index++;
	}

	if (dummy_latency_index == max_latency_table_entries) {
		ASSERT(dummy_latency_index != max_latency_table_entries);
		/* If the execution gets here, it means dummy p_states are
		 * not possible. This should never happen and would mean
		 * something is severely wrong.
		 * Here we reset dummy_latency_index to 3, because it is
		 * better to have underflows than system crashes.
		 */
		dummy_latency_index = max_latency_table_entries - 1;
	}

	return dummy_latency_index;
}

/**
 * dcn32_helper_populate_phantom_dlg_params - Get DLG params for phantom pipes
 * and populate pipe_ctx with those params.
 * @dc: [in] current dc state
 * @context: [in] new dc state
 * @pipes: [in] DML pipe params array
 * @pipe_cnt: [in] DML pipe count
 *
 * This function must be called AFTER the phantom pipes are added to context
 * and run through DML (so that the DLG params for the phantom pipes can be
 * populated), and BEFORE we program the timing for the phantom pipes.
 */
void dcn32_helper_populate_phantom_dlg_params(struct dc *dc,
					      struct dc_state *context,
					      display_e2e_pipe_params_st *pipes,
					      int pipe_cnt)
{
	uint32_t i, pipe_idx;

	dc_assert_fp_enabled();

	for (i = 0, pipe_idx = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];

		if (!pipe->stream)
			continue;

		if (pipe->plane_state && dc_state_get_pipe_subvp_type(context, pipe) == SUBVP_PHANTOM) {
			pipes[pipe_idx].pipe.dest.vstartup_start =
				get_vstartup(&context->bw_ctx.dml, pipes, pipe_cnt, pipe_idx);
			pipes[pipe_idx].pipe.dest.vupdate_offset =
				get_vupdate_offset(&context->bw_ctx.dml, pipes, pipe_cnt, pipe_idx);
			pipes[pipe_idx].pipe.dest.vupdate_width =
				get_vupdate_width(&context->bw_ctx.dml, pipes, pipe_cnt, pipe_idx);
			pipes[pipe_idx].pipe.dest.vready_offset =
				get_vready_offset(&context->bw_ctx.dml, pipes, pipe_cnt, pipe_idx);
			pipe->pipe_dlg_param = pipes[pipe_idx].pipe.dest;
		}
		pipe_idx++;
	}
}

static float calculate_net_bw_in_kbytes_sec(struct _vcs_dpi_voltage_scaling_st *entry)
{
	float memory_bw_kbytes_sec;
	float fabric_bw_kbytes_sec;
	float sdp_bw_kbytes_sec;
	float limiting_bw_kbytes_sec;

	memory_bw_kbytes_sec = entry->dram_speed_mts *
				dcn3_2_soc.num_chans *
				dcn3_2_soc.dram_channel_width_bytes *
				((float)dcn3_2_soc.pct_ideal_dram_sdp_bw_after_urgent_pixel_only / 100);

	fabric_bw_kbytes_sec = entry->fabricclk_mhz *
				dcn3_2_soc.return_bus_width_bytes *
				((float)dcn3_2_soc.pct_ideal_fabric_bw_after_urgent / 100);

	sdp_bw_kbytes_sec = entry->dcfclk_mhz *
				dcn3_2_soc.return_bus_width_bytes *
				((float)dcn3_2_soc.pct_ideal_sdp_bw_after_urgent / 100);

	limiting_bw_kbytes_sec = memory_bw_kbytes_sec;

	if (fabric_bw_kbytes_sec < limiting_bw_kbytes_sec)
		limiting_bw_kbytes_sec = fabric_bw_kbytes_sec;

	if (sdp_bw_kbytes_sec < limiting_bw_kbytes_sec)
		limiting_bw_kbytes_sec = sdp_bw_kbytes_sec;

	return limiting_bw_kbytes_sec;
}

static void get_optimal_ntuple(struct _vcs_dpi_voltage_scaling_st *entry)
{
	if (entry->dcfclk_mhz > 0) {
		float bw_on_sdp = entry->dcfclk_mhz * dcn3_2_soc.return_bus_width_bytes * ((float)dcn3_2_soc.pct_ideal_sdp_bw_after_urgent / 100);

		entry->fabricclk_mhz = bw_on_sdp / (dcn3_2_soc.return_bus_width_bytes * ((float)dcn3_2_soc.pct_ideal_fabric_bw_after_urgent / 100));
		entry->dram_speed_mts = bw_on_sdp / (dcn3_2_soc.num_chans *
				dcn3_2_soc.dram_channel_width_bytes * ((float)dcn3_2_soc.pct_ideal_dram_sdp_bw_after_urgent_pixel_only / 100));
	} else if (entry->fabricclk_mhz > 0) {
		float bw_on_fabric = entry->fabricclk_mhz * dcn3_2_soc.return_bus_width_bytes * ((float)dcn3_2_soc.pct_ideal_fabric_bw_after_urgent / 100);

		entry->dcfclk_mhz = bw_on_fabric / (dcn3_2_soc.return_bus_width_bytes * ((float)dcn3_2_soc.pct_ideal_sdp_bw_after_urgent / 100));
		entry->dram_speed_mts = bw_on_fabric / (dcn3_2_soc.num_chans *
				dcn3_2_soc.dram_channel_width_bytes * ((float)dcn3_2_soc.pct_ideal_dram_sdp_bw_after_urgent_pixel_only / 100));
	} else if (entry->dram_speed_mts > 0) {
		float bw_on_dram = entry->dram_speed_mts * dcn3_2_soc.num_chans *
				dcn3_2_soc.dram_channel_width_bytes * ((float)dcn3_2_soc.pct_ideal_dram_sdp_bw_after_urgent_pixel_only / 100);

		entry->fabricclk_mhz = bw_on_dram / (dcn3_2_soc.return_bus_width_bytes * ((float)dcn3_2_soc.pct_ideal_fabric_bw_after_urgent / 100));
		entry->dcfclk_mhz = bw_on_dram / (dcn3_2_soc.return_bus_width_bytes * ((float)dcn3_2_soc.pct_ideal_sdp_bw_after_urgent / 100));
	}
}

static void insert_entry_into_table_sorted(struct _vcs_dpi_voltage_scaling_st *table,
				    unsigned int *num_entries,
				    struct _vcs_dpi_voltage_scaling_st *entry)
{
	int i = 0;
	int index = 0;

	dc_assert_fp_enabled();

	if (*num_entries == 0) {
		table[0] = *entry;
		(*num_entries)++;
	} else {
		while (entry->net_bw_in_kbytes_sec > table[index].net_bw_in_kbytes_sec) {
			index++;
			if (index >= *num_entries)
				break;
		}

		for (i = *num_entries; i > index; i--)
			table[i] = table[i - 1];

		table[index] = *entry;
		(*num_entries)++;
	}
}

/**
 * dcn32_set_phantom_stream_timing - Set timing params for the phantom stream
 * @dc: current dc state
 * @context: new dc state
 * @ref_pipe: Main pipe for the phantom stream
 * @phantom_stream: target phantom stream state
 * @pipes: DML pipe params
 * @pipe_cnt: number of DML pipes
 * @dc_pipe_idx: DC pipe index for the main pipe (i.e. ref_pipe)
 *
 * Set timing params of the phantom stream based on calculated output from DML.
 * This function first gets the DML pipe index using the DC pipe index, then
 * calls into DML (get_subviewport_lines_needed_in_mall) to get the number of
 * lines required for SubVP MCLK switching and assigns to the phantom stream
 * accordingly.
 *
 * - The number of SubVP lines calculated in DML does not take into account
 * FW processing delays and required pstate allow width, so we must include
 * that separately.
 *
 * - Set phantom backporch = vstartup of main pipe
 */
void dcn32_set_phantom_stream_timing(struct dc *dc,
				     struct dc_state *context,
				     struct pipe_ctx *ref_pipe,
				     struct dc_stream_state *phantom_stream,
				     display_e2e_pipe_params_st *pipes,
				     unsigned int pipe_cnt,
				     unsigned int dc_pipe_idx)
{
	unsigned int i, pipe_idx;
	struct pipe_ctx *pipe;
	uint32_t phantom_vactive, phantom_bp, pstate_width_fw_delay_lines;
	unsigned int num_dpp;
	unsigned int vlevel = context->bw_ctx.dml.vba.VoltageLevel;
	unsigned int dcfclk = context->bw_ctx.dml.vba.DCFCLKState[vlevel][context->bw_ctx.dml.vba.maxMpcComb];
	unsigned int socclk = context->bw_ctx.dml.vba.SOCCLKPerState[vlevel];
	struct vba_vars_st *vba = &context->bw_ctx.dml.vba;
	struct dc_stream_state *main_stream = ref_pipe->stream;

	dc_assert_fp_enabled();

	// Find DML pipe index (pipe_idx) using dc_pipe_idx
	for (i = 0, pipe_idx = 0; i < dc->res_pool->pipe_count; i++) {
		pipe = &context->res_ctx.pipe_ctx[i];

		if (!pipe->stream)
			continue;

		if (i == dc_pipe_idx)
			break;

		pipe_idx++;
	}

	// Calculate lines required for pstate allow width and FW processing delays
	pstate_width_fw_delay_lines = ((double)(dc->caps.subvp_fw_processing_delay_us +
			dc->caps.subvp_pstate_allow_width_us) / 1000000) *
			(ref_pipe->stream->timing.pix_clk_100hz * 100) /
			(double)ref_pipe->stream->timing.h_total;

	// Update clks_cfg for calling into recalculate
	pipes[0].clks_cfg.voltage = vlevel;
	pipes[0].clks_cfg.dcfclk_mhz = dcfclk;
	pipes[0].clks_cfg.socclk_mhz = socclk;

	// DML calculation for MALL region doesn't take into account FW delay
	// and required pstate allow width for multi-display cases
	/* Add 16 lines margin to the MALL REGION because SUB_VP_START_LINE must be aligned
	 * to 2 swaths (i.e. 16 lines)
	 */
	phantom_vactive = get_subviewport_lines_needed_in_mall(&context->bw_ctx.dml, pipes, pipe_cnt, pipe_idx) +
				pstate_width_fw_delay_lines + dc->caps.subvp_swath_height_margin_lines;

	// W/A for DCC corruption with certain high resolution timings.
	// Determing if pipesplit is used. If so, add meta_row_height to the phantom vactive.
	num_dpp = vba->NoOfDPP[vba->VoltageLevel][vba->maxMpcComb][vba->pipe_plane[pipe_idx]];
	phantom_vactive += num_dpp > 1 ? vba->meta_row_height[vba->pipe_plane[pipe_idx]] : 0;

	/* dc->debug.subvp_extra_lines 0 by default*/
	phantom_vactive += dc->debug.subvp_extra_lines;

	// For backporch of phantom pipe, use vstartup of the main pipe
	phantom_bp = get_vstartup(&context->bw_ctx.dml, pipes, pipe_cnt, pipe_idx);

	phantom_stream->dst.y = 0;
	phantom_stream->dst.height = phantom_vactive;
	/* When scaling, DML provides the end to end required number of lines for MALL.
	 * dst.height is always correct for this case, but src.height is not which causes a
	 * delta between main and phantom pipe scaling outputs. Need to adjust src.height on
	 * phantom for this case.
	 */
	phantom_stream->src.y = 0;
	phantom_stream->src.height = (double)phantom_vactive * (double)main_stream->src.height / (double)main_stream->dst.height;

	phantom_stream->timing.v_addressable = phantom_vactive;
	phantom_stream->timing.v_front_porch = 1;
	phantom_stream->timing.v_total = phantom_stream->timing.v_addressable +
						phantom_stream->timing.v_front_porch +
						phantom_stream->timing.v_sync_width +
						phantom_bp;
	phantom_stream->timing.flags.DSC = 0; // Don't need DSC for phantom timing
}

/**
 * dcn32_get_num_free_pipes - Calculate number of free pipes
 * @dc: current dc state
 * @context: new dc state
 *
 * This function assumes that a "used" pipe is a pipe that has
 * both a stream and a plane assigned to it.
 *
 * Return: Number of free pipes available in the context
 */
static unsigned int dcn32_get_num_free_pipes(struct dc *dc, struct dc_state *context)
{
	unsigned int i;
	unsigned int free_pipes = 0;
	unsigned int num_pipes = 0;

	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];

		if (pipe->stream && !pipe->top_pipe) {
			while (pipe) {
				num_pipes++;
				pipe = pipe->bottom_pipe;
			}
		}
	}

	free_pipes = dc->res_pool->pipe_count - num_pipes;
	return free_pipes;
}

/**
 * dcn32_assign_subvp_pipe - Function to decide which pipe will use Sub-VP.
 * @dc: current dc state
 * @context: new dc state
 * @index: [out] dc pipe index for the pipe chosen to have phantom pipes assigned
 *
 * We enter this function if we are Sub-VP capable (i.e. enough pipes available)
 * and regular P-State switching (i.e. VACTIVE/VBLANK) is not supported, or if
 * we are forcing SubVP P-State switching on the current config.
 *
 * The number of pipes used for the chosen surface must be less than or equal to the
 * number of free pipes available.
 *
 * In general we choose surfaces with the longest frame time first (better for SubVP + VBLANK).
 * For multi-display cases the ActiveDRAMClockChangeMargin doesn't provide enough info on its own
 * for determining which should be the SubVP pipe (need a way to determine if a pipe / plane doesn't
 * support MCLK switching naturally [i.e. ACTIVE or VBLANK]).
 *
 * Return: True if a valid pipe assignment was found for Sub-VP. Otherwise false.
 */
static bool dcn32_assign_subvp_pipe(struct dc *dc,
				    struct dc_state *context,
				    unsigned int *index)
{
	unsigned int i, pipe_idx;
	unsigned int max_frame_time = 0;
	bool valid_assignment_found = false;
	unsigned int free_pipes = dcn32_get_num_free_pipes(dc, context);
	struct vba_vars_st *vba = &context->bw_ctx.dml.vba;

	for (i = 0, pipe_idx = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];
		unsigned int num_pipes = 0;
		unsigned int refresh_rate = 0;

		if (!pipe->stream)
			continue;

		// Round up
		refresh_rate = (pipe->stream->timing.pix_clk_100hz * 100 +
				pipe->stream->timing.v_total * pipe->stream->timing.h_total - 1)
				/ (double)(pipe->stream->timing.v_total * pipe->stream->timing.h_total);
		/* SubVP pipe candidate requirements:
		 * - Refresh rate < 120hz
		 * - Not able to switch in vactive naturally (switching in active means the
		 *   DET provides enough buffer to hide the P-State switch latency -- trying
		 *   to combine this with SubVP can cause issues with the scheduling).
		 * - Not TMZ surface
		 */
		if (pipe->plane_state && !pipe->top_pipe && !pipe->prev_odm_pipe && !dcn32_is_center_timing(pipe) &&
				!(pipe->stream->timing.pix_clk_100hz / 10000 > DCN3_2_MAX_SUBVP_PIXEL_RATE_MHZ) &&
				(!dcn32_is_psr_capable(pipe) || (context->stream_count == 1 && dc->caps.dmub_caps.subvp_psr)) &&
				dc_state_get_pipe_subvp_type(context, pipe) == SUBVP_NONE &&
				(refresh_rate < 120 || dcn32_allow_subvp_high_refresh_rate(dc, context, pipe)) &&
				!pipe->plane_state->address.tmz_surface &&
				(vba->ActiveDRAMClockChangeLatencyMarginPerState[vba->VoltageLevel][vba->maxMpcComb][vba->pipe_plane[pipe_idx]] <= 0 ||
				(vba->ActiveDRAMClockChangeLatencyMarginPerState[vba->VoltageLevel][vba->maxMpcComb][vba->pipe_plane[pipe_idx]] > 0 &&
						dcn32_allow_subvp_with_active_margin(pipe)))) {
			while (pipe) {
				num_pipes++;
				pipe = pipe->bottom_pipe;
			}

			pipe = &context->res_ctx.pipe_ctx[i];
			if (num_pipes <= free_pipes) {
				struct dc_stream_state *stream = pipe->stream;
				unsigned int frame_us = (stream->timing.v_total * stream->timing.h_total /
						(double)(stream->timing.pix_clk_100hz * 100)) * 1000000;
				if (frame_us > max_frame_time) {
					*index = i;
					max_frame_time = frame_us;
					valid_assignment_found = true;
				}
			}
		}
		pipe_idx++;
	}
	return valid_assignment_found;
}

/**
 * dcn32_enough_pipes_for_subvp - Function to check if there are "enough" pipes for SubVP.
 * @dc: current dc state
 * @context: new dc state
 *
 * This function returns true if there are enough free pipes
 * to create the required phantom pipes for any given stream
 * (that does not already have phantom pipe assigned).
 *
 * e.g. For a 2 stream config where the first stream uses one
 * pipe and the second stream uses 2 pipes (i.e. pipe split),
 * this function will return true because there is 1 remaining
 * pipe which can be used as the phantom pipe for the non pipe
 * split pipe.
 *
 * Return:
 * True if there are enough free pipes to assign phantom pipes to at least one
 * stream that does not already have phantom pipes assigned. Otherwise false.
 */
static bool dcn32_enough_pipes_for_subvp(struct dc *dc, struct dc_state *context)
{
	unsigned int i, split_cnt, free_pipes;
	unsigned int min_pipe_split = dc->res_pool->pipe_count + 1; // init as max number of pipes + 1
	bool subvp_possible = false;

	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];

		// Find the minimum pipe split count for non SubVP pipes
		if (resource_is_pipe_type(pipe, OPP_HEAD) &&
			dc_state_get_pipe_subvp_type(context, pipe) == SUBVP_NONE) {
			split_cnt = 0;
			while (pipe) {
				split_cnt++;
				pipe = pipe->bottom_pipe;
			}

			if (split_cnt < min_pipe_split)
				min_pipe_split = split_cnt;
		}
	}

	free_pipes = dcn32_get_num_free_pipes(dc, context);

	// SubVP only possible if at least one pipe is being used (i.e. free_pipes
	// should not equal to the pipe_count)
	if (free_pipes >= min_pipe_split && free_pipes < dc->res_pool->pipe_count)
		subvp_possible = true;

	return subvp_possible;
}

/**
 * subvp_subvp_schedulable - Determine if SubVP + SubVP config is schedulable
 * @dc: current dc state
 * @context: new dc state
 *
 * High level algorithm:
 * 1. Find longest microschedule length (in us) between the two SubVP pipes
 * 2. Check if the worst case overlap (VBLANK in middle of ACTIVE) for both
 * pipes still allows for the maximum microschedule to fit in the active
 * region for both pipes.
 *
 * Return: True if the SubVP + SubVP config is schedulable, false otherwise
 */
static bool subvp_subvp_schedulable(struct dc *dc, struct dc_state *context)
{
	struct pipe_ctx *subvp_pipes[2] = {0};
	struct dc_stream_state *phantom = NULL;
	uint32_t microschedule_lines = 0;
	uint32_t index = 0;
	uint32_t i;
	uint32_t max_microschedule_us = 0;
	int32_t vactive1_us, vactive2_us, vblank1_us, vblank2_us;

	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];
		uint32_t time_us = 0;

		/* Loop to calculate the maximum microschedule time between the two SubVP pipes,
		 * and also to store the two main SubVP pipe pointers in subvp_pipes[2].
		 */
		phantom = dc_state_get_paired_subvp_stream(context, pipe->stream);
		if (phantom && pipe->stream && pipe->plane_state && !pipe->top_pipe &&
			dc_state_get_pipe_subvp_type(context, pipe) == SUBVP_MAIN) {
			microschedule_lines = (phantom->timing.v_total - phantom->timing.v_front_porch) +
					phantom->timing.v_addressable;

			// Round up when calculating microschedule time (+ 1 at the end)
			time_us = (microschedule_lines * phantom->timing.h_total) /
					(double)(phantom->timing.pix_clk_100hz * 100) * 1000000 +
						dc->caps.subvp_prefetch_end_to_mall_start_us +
						dc->caps.subvp_fw_processing_delay_us + 1;
			if (time_us > max_microschedule_us)
				max_microschedule_us = time_us;

			subvp_pipes[index] = pipe;
			index++;

			// Maximum 2 SubVP pipes
			if (index == 2)
				break;
		}
	}
	vactive1_us = ((subvp_pipes[0]->stream->timing.v_addressable * subvp_pipes[0]->stream->timing.h_total) /
			(double)(subvp_pipes[0]->stream->timing.pix_clk_100hz * 100)) * 1000000;
	vactive2_us = ((subvp_pipes[1]->stream->timing.v_addressable * subvp_pipes[1]->stream->timing.h_total) /
				(double)(subvp_pipes[1]->stream->timing.pix_clk_100hz * 100)) * 1000000;
	vblank1_us = (((subvp_pipes[0]->stream->timing.v_total - subvp_pipes[0]->stream->timing.v_addressable) *
			subvp_pipes[0]->stream->timing.h_total) /
			(double)(subvp_pipes[0]->stream->timing.pix_clk_100hz * 100)) * 1000000;
	vblank2_us = (((subvp_pipes[1]->stream->timing.v_total - subvp_pipes[1]->stream->timing.v_addressable) *
			subvp_pipes[1]->stream->timing.h_total) /
			(double)(subvp_pipes[1]->stream->timing.pix_clk_100hz * 100)) * 1000000;

	if ((vactive1_us - vblank2_us) / 2 > max_microschedule_us &&
	    (vactive2_us - vblank1_us) / 2 > max_microschedule_us)
		return true;

	return false;
}

/**
 * subvp_drr_schedulable() - Determine if SubVP + DRR config is schedulable
 * @dc: current dc state
 * @context: new dc state
 *
 * High level algorithm:
 * 1. Get timing for SubVP pipe, phantom pipe, and DRR pipe
 * 2. Determine the frame time for the DRR display when adding required margin for MCLK switching
 * (the margin is equal to the MALL region + DRR margin (500us))
 * 3.If (SubVP Active - Prefetch > Stretched DRR frame + max(MALL region, Stretched DRR frame))
 * then report the configuration as supported
 *
 * Return: True if the SubVP + DRR config is schedulable, false otherwise
 */
static bool subvp_drr_schedulable(struct dc *dc, struct dc_state *context)
{
	bool schedulable = false;
	uint32_t i;
	struct pipe_ctx *pipe = NULL;
	struct pipe_ctx *drr_pipe = NULL;
	struct dc_crtc_timing *main_timing = NULL;
	struct dc_crtc_timing *phantom_timing = NULL;
	struct dc_crtc_timing *drr_timing = NULL;
	int16_t prefetch_us = 0;
	int16_t mall_region_us = 0;
	int16_t drr_frame_us = 0;	// nominal frame time
	int16_t subvp_active_us = 0;
	int16_t stretched_drr_us = 0;
	int16_t drr_stretched_vblank_us = 0;
	int16_t max_vblank_mallregion = 0;
	struct dc_stream_state *phantom_stream;
	bool subvp_found = false;
	bool drr_found = false;

	// Find SubVP pipe
	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		pipe = &context->res_ctx.pipe_ctx[i];

		// We check for master pipe, but it shouldn't matter since we only need
		// the pipe for timing info (stream should be same for any pipe splits)
		if (!resource_is_pipe_type(pipe, OTG_MASTER) ||
				!resource_is_pipe_type(pipe, DPP_PIPE))
			continue;

		// Find the SubVP pipe
		if (dc_state_get_pipe_subvp_type(context, pipe) == SUBVP_MAIN) {
			subvp_found = true;
			break;
		}
	}

	// Find the DRR pipe
	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		drr_pipe = &context->res_ctx.pipe_ctx[i];

		// We check for master pipe only
		if (!resource_is_pipe_type(drr_pipe, OTG_MASTER) ||
				!resource_is_pipe_type(drr_pipe, DPP_PIPE))
			continue;

		if (dc_state_get_pipe_subvp_type(context, drr_pipe) == SUBVP_NONE && drr_pipe->stream->ignore_msa_timing_param &&
				(drr_pipe->stream->allow_freesync || drr_pipe->stream->vrr_active_variable || drr_pipe->stream->vrr_active_fixed)) {
			drr_found = true;
			break;
		}
	}

	phantom_stream = dc_state_get_paired_subvp_stream(context, pipe->stream);
	if (phantom_stream && subvp_found && drr_found) {
		main_timing = &pipe->stream->timing;
		phantom_timing = &phantom_stream->timing;
		drr_timing = &drr_pipe->stream->timing;
		prefetch_us = (phantom_timing->v_total - phantom_timing->v_front_porch) * phantom_timing->h_total /
				(double)(phantom_timing->pix_clk_100hz * 100) * 1000000 +
				dc->caps.subvp_prefetch_end_to_mall_start_us;
		subvp_active_us = main_timing->v_addressable * main_timing->h_total /
				(double)(main_timing->pix_clk_100hz * 100) * 1000000;
		drr_frame_us = drr_timing->v_total * drr_timing->h_total /
				(double)(drr_timing->pix_clk_100hz * 100) * 1000000;
		// P-State allow width and FW delays already included phantom_timing->v_addressable
		mall_region_us = phantom_timing->v_addressable * phantom_timing->h_total /
				(double)(phantom_timing->pix_clk_100hz * 100) * 1000000;
		stretched_drr_us = drr_frame_us + mall_region_us + SUBVP_DRR_MARGIN_US;
		drr_stretched_vblank_us = (drr_timing->v_total - drr_timing->v_addressable) * drr_timing->h_total /
				(double)(drr_timing->pix_clk_100hz * 100) * 1000000 + (stretched_drr_us - drr_frame_us);
		max_vblank_mallregion = drr_stretched_vblank_us > mall_region_us ? drr_stretched_vblank_us : mall_region_us;
	}

	/* We consider SubVP + DRR schedulable if the stretched frame duration of the DRR display (i.e. the
	 * highest refresh rate + margin that can support UCLK P-State switch) passes the static analysis
	 * for VBLANK: (VACTIVE region of the SubVP pipe can fit the MALL prefetch, VBLANK frame time,
	 * and the max of (VBLANK blanking time, MALL region)).
	 */
	if (drr_timing &&
	    stretched_drr_us < (1 / (double)drr_timing->min_refresh_in_uhz) * 1000000 * 1000000 &&
	    subvp_active_us - prefetch_us - stretched_drr_us - max_vblank_mallregion > 0)
		schedulable = true;

	return schedulable;
}


/**
 * subvp_vblank_schedulable - Determine if SubVP + VBLANK config is schedulable
 * @dc: current dc state
 * @context: new dc state
 *
 * High level algorithm:
 * 1. Get timing for SubVP pipe, phantom pipe, and VBLANK pipe
 * 2. If (SubVP Active - Prefetch > Vblank Frame Time + max(MALL region, Vblank blanking time))
 * then report the configuration as supported
 * 3. If the VBLANK display is DRR, then take the DRR static schedulability path
 *
 * Return: True if the SubVP + VBLANK/DRR config is schedulable, false otherwise
 */
static bool subvp_vblank_schedulable(struct dc *dc, struct dc_state *context)
{
	struct pipe_ctx *pipe = NULL;
	struct pipe_ctx *subvp_pipe = NULL;
	bool found = false;
	bool schedulable = false;
	uint32_t i = 0;
	uint8_t vblank_index = 0;
	uint16_t prefetch_us = 0;
	uint16_t mall_region_us = 0;
	uint16_t vblank_frame_us = 0;
	uint16_t subvp_active_us = 0;
	uint16_t vblank_blank_us = 0;
	uint16_t max_vblank_mallregion = 0;
	struct dc_crtc_timing *main_timing = NULL;
	struct dc_crtc_timing *phantom_timing = NULL;
	struct dc_crtc_timing *vblank_timing = NULL;
	struct dc_stream_state *phantom_stream;
	enum mall_stream_type pipe_mall_type;

	/* For SubVP + VBLANK/DRR cases, we assume there can only be
	 * a single VBLANK/DRR display. If DML outputs SubVP + VBLANK
	 * is supported, it is either a single VBLANK case or two VBLANK
	 * displays which are synchronized (in which case they have identical
	 * timings).
	 */
	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		pipe = &context->res_ctx.pipe_ctx[i];
		pipe_mall_type = dc_state_get_pipe_subvp_type(context, pipe);

		// We check for master pipe, but it shouldn't matter since we only need
		// the pipe for timing info (stream should be same for any pipe splits)
		if (!resource_is_pipe_type(pipe, OTG_MASTER) ||
				!resource_is_pipe_type(pipe, DPP_PIPE))
			continue;

		if (!found && pipe_mall_type == SUBVP_NONE) {
			// Found pipe which is not SubVP or Phantom (i.e. the VBLANK pipe).
			vblank_index = i;
			found = true;
		}

		if (!subvp_pipe && pipe_mall_type == SUBVP_MAIN)
			subvp_pipe = pipe;
	}
	if (found && subvp_pipe) {
		phantom_stream = dc_state_get_paired_subvp_stream(context, subvp_pipe->stream);
		main_timing = &subvp_pipe->stream->timing;
		phantom_timing = &phantom_stream->timing;
		vblank_timing = &context->res_ctx.pipe_ctx[vblank_index].stream->timing;
		// Prefetch time is equal to VACTIVE + BP + VSYNC of the phantom pipe
		// Also include the prefetch end to mallstart delay time
		prefetch_us = (phantom_timing->v_total - phantom_timing->v_front_porch) * phantom_timing->h_total /
				(double)(phantom_timing->pix_clk_100hz * 100) * 1000000 +
				dc->caps.subvp_prefetch_end_to_mall_start_us;
		// P-State allow width and FW delays already included phantom_timing->v_addressable
		mall_region_us = phantom_timing->v_addressable * phantom_timing->h_total /
				(double)(phantom_timing->pix_clk_100hz * 100) * 1000000;
		vblank_frame_us = vblank_timing->v_total * vblank_timing->h_total /
				(double)(vblank_timing->pix_clk_100hz * 100) * 1000000;
		vblank_blank_us =  (vblank_timing->v_total - vblank_timing->v_addressable) * vblank_timing->h_total /
				(double)(vblank_timing->pix_clk_100hz * 100) * 1000000;
		subvp_active_us = main_timing->v_addressable * main_timing->h_total /
				(double)(main_timing->pix_clk_100hz * 100) * 1000000;
		max_vblank_mallregion = vblank_blank_us > mall_region_us ? vblank_blank_us : mall_region_us;

		// Schedulable if VACTIVE region of the SubVP pipe can fit the MALL prefetch, VBLANK frame time,
		// and the max of (VBLANK blanking time, MALL region)
		// TODO: Possibly add some margin (i.e. the below conditions should be [...] > X instead of [...] > 0)
		if (subvp_active_us - prefetch_us - vblank_frame_us - max_vblank_mallregion > 0)
			schedulable = true;
	}
	return schedulable;
}

/**
 * subvp_subvp_admissable() - Determine if subvp + subvp config is admissible
 *
 * @dc: Current DC state
 * @context: New DC state to be programmed
 *
 * SubVP + SubVP is admissible under the following conditions:
 * - All SubVP pipes are < 120Hz OR
 * - All SubVP pipes are >= 120hz
 *
 * Return: True if admissible, false otherwise
 */
static bool subvp_subvp_admissable(struct dc *dc,
				struct dc_state *context)
{
	bool result = false;
	uint32_t i;
	uint8_t subvp_count = 0;
	uint32_t min_refresh = subvp_high_refresh_list.min_refresh, max_refresh = 0;
	uint64_t refresh_rate = 0;

	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];

		if (!pipe->stream)
			continue;

		if (pipe->plane_state && !pipe->top_pipe &&
				dc_state_get_pipe_subvp_type(context, pipe) == SUBVP_MAIN) {
			refresh_rate = (pipe->stream->timing.pix_clk_100hz * (uint64_t)100 +
				pipe->stream->timing.v_total * (uint64_t)pipe->stream->timing.h_total - (uint64_t)1);
			refresh_rate = div_u64(refresh_rate, pipe->stream->timing.v_total);
			refresh_rate = div_u64(refresh_rate, pipe->stream->timing.h_total);

			if ((uint32_t)refresh_rate < min_refresh)
				min_refresh = (uint32_t)refresh_rate;
			if ((uint32_t)refresh_rate > max_refresh)
				max_refresh = (uint32_t)refresh_rate;
			subvp_count++;
		}
	}

	if (subvp_count == 2 && ((min_refresh < 120 && max_refresh < 120) ||
		(min_refresh >= subvp_high_refresh_list.min_refresh &&
				max_refresh <= subvp_high_refresh_list.max_refresh)))
		result = true;

	return result;
}

/**
 * subvp_validate_static_schedulability - Check which SubVP case is calculated
 * and handle static analysis based on the case.
 * @dc: current dc state
 * @context: new dc state
 * @vlevel: Voltage level calculated by DML
 *
 * Three cases:
 * 1. SubVP + SubVP
 * 2. SubVP + VBLANK (DRR checked internally)
 * 3. SubVP + VACTIVE (currently unsupported)
 *
 * Return: True if statically schedulable, false otherwise
 */
static bool subvp_validate_static_schedulability(struct dc *dc,
				struct dc_state *context,
				int vlevel)
{
	bool schedulable = false;
	struct vba_vars_st *vba = &context->bw_ctx.dml.vba;
	uint32_t i, pipe_idx;
	uint8_t subvp_count = 0;
	uint8_t vactive_count = 0;
	uint8_t non_subvp_pipes = 0;

	for (i = 0, pipe_idx = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe = &context->res_ctx.pipe_ctx[i];
		enum mall_stream_type pipe_mall_type = dc_state_get_pipe_subvp_type(context, pipe);

		if (!pipe->stream)
			continue;

		if (pipe->plane_state && !pipe->top_pipe) {
			if (pipe_mall_type == SUBVP_MAIN)
				subvp_count++;
			if (pipe_mall_type == SUBVP_NONE)
				non_subvp_pipes++;
		}

		// Count how many planes that aren't SubVP/phantom are capable of VACTIVE
		// switching (SubVP + VACTIVE unsupported). In situations where we force
		// SubVP for a VACTIVE plane, we don't want to increment the vactive_count.
		if (vba->ActiveDRAMClockChangeLatencyMarginPerState[vlevel][vba->maxMpcComb][vba->pipe_plane[pipe_idx]] > 0 &&
				pipe_mall_type == SUBVP_NONE) {
			vactive_count++;
		}
		pipe_idx++;
	}

	if (subvp_count == 2) {
		// Static schedulability check for SubVP + SubVP case
		schedulable = subvp_subvp_admissable(dc, context) && subvp_subvp_schedulable(dc, context);
	} else if (subvp_count == 1 && non_subvp_pipes == 0) {
		// Single SubVP configs will be supported by default as long as it's suppported by DML
		schedulable = true;
	} else if (subvp_count == 1 && non_subvp_pipes == 1) {
		if (dcn32_subvp_drr_admissable(dc, context))
			schedulable = subvp_drr_schedulable(dc, context);
		else if (dcn32_subvp_vblank_admissable(dc, context, vlevel))
			schedulable = subvp_vblank_schedulable(dc, context);
	} else if (vba->DRAMClockChangeSupport[vlevel][vba->maxMpcComb] == dm_dram_clock_change_vactive_w_mall_sub_vp &&
			vactive_count > 0) {
		// For single display SubVP cases, DML will output dm_dram_clock_change_vactive_w_mall_sub_vp by default.
		// We tell the difference between SubVP vs. SubVP + VACTIVE by checking the vactive_count.
		// SubVP + VACTIVE currently unsupported
		schedulable = false;
	}
	return schedulable;
}

static void assign_subvp_index(struct dc *dc, struct dc_state *context)
{
	int i;
	int index = 0;

	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		struct pipe_ctx *pipe_ctx = &context->res_ctx.pipe_ctx[i];

		if (resource_is_pipe_type(pipe_ctx, OTG_MASTER) &&
				dc_state_get_pipe_subvp_type(context, pipe_ctx) == SUBVP_MAIN) {
			pipe_ctx->subvp_index = index++;
		} else {
			pipe_ctx->subvp_index = 0;
		}
	}
}

struct pipe_slice_table {
	struct {
		struct dc_stream_state *stream;
		int slice_count;
	} odm_combines[MAX_STREAMS];
	int odm_combine_count;

	struct {
		struct pipe_ctx *pri_pipe;
		struct dc_plane_state *plane;
		int slice_count;
	} mpc_combines[MAX_PLANES];
	int mpc_combine_count;
};


static void update_slice_table_for_stream(struct pipe_slice_table *table,
		struct dc_stream_state *stream, int diff)
{
	int i;

	for (i = 0; i < table->odm_combine_count; i++) {
		if (table->odm_combines[i].stream == stream) {
			table->odm_combines[i].slice_count += diff;
			break;
		}
	}

	if (i == table->odm_combine_count) {
		table->odm_combine_count++;
		table->odm_combines[i].stream = stream;
		table->odm_combines[i].slice_count = diff;
	}
}

static void update_slice_table_for_plane(struct pipe_slice_table *table,
		struct pipe_ctx *dpp_pipe, struct dc_plane_state *plane, int diff)
{
	int i;
	struct pipe_ctx *pri_dpp_pipe = resource_get_primary_dpp_pipe(dpp_pipe);

	for (i = 0; i < table->mpc_combine_count; i++) {
		if (table->mpc_combines[i].plane == plane &&
				table->mpc_combines[i].pri_pipe == pri_dpp_pipe) {
			table->mpc_combines[i].slice_count += diff;
			break;
		}
	}

	if (i == table->mpc_combine_count) {
		table->mpc_combine_count++;
		table->mpc_combines[i].plane = plane;
		table->mpc_combines[i].pri_pipe = pri_dpp_pipe;
		table->mpc_combines[i].slice_count = diff;
	}
}

static void init_pipe_slice_table_from_context(
		struct pipe_slice_table *table,
		struct dc_state *context)
{
	int i, j;
	struct pipe_ctx *otg_master;
	struct pipe_ctx *dpp_pipes[MAX_PIPES];
	struct dc_stream_state *stream;
	int count;

	memset(table, 0, sizeof(*table));

	for (i = 0; i < context->stream_count; i++) {
		stream = context->streams[i];
		otg_master = resource_get_otg_master_for_stream(
				&context->res_ctx, stream);
		if (!otg_master)
			continue;

		count = resource_get_odm_slice_count(otg_master);
		update_slice_table_for_stream(table, stream, count);

		count = resource_get_dpp_pipes_for_opp_head(otg_master,
				&context->res_ctx, dpp_pipes);
		for (j = 0; j < count; j++)
			if (dpp_pipes[j]->plane_state)
				update_slice_table_for_plane(table, dpp_pipes[j],
						dpp_pipes[j]->plane_state, 1);
	}
}

static bool update_pipe_slice_table_with_split_flags(
		struct pipe_slice_table *table,
		struct dc *dc,
		struct dc_state *context,
		struct vba_vars_st *vba,
		int split[MAX_PIPES],
		bool merge[MAX_PIPES])
{
	/* NOTE: we are deprecating the support for the concept of pipe splitting
	 * or pipe merging. Instead we append slices to the end and remove
	 * slices from the end. The following code converts a pipe split or
	 * merge to an append or remove operation.
	 *
	 * For example:
	 * When split flags describe the following pipe connection transition
	 *
	 * from:
	 *  pipe 0 (split=2) -> pipe 1 (split=2)
	 * to: (old behavior)
	 *  pipe 0 -> pipe 2 -> pipe 1 -> pipe 3
	 *
	 * the code below actually does:
	 *  pipe 0 -> pipe 1 -> pipe 2 -> pipe 3
	 *
	 * This is the new intended behavior and for future DCNs we will retire
	 * the old concept completely.
	 */
	struct pipe_ctx *pipe;
	bool odm;
	int dc_pipe_idx, dml_pipe_idx = 0;
	bool updated = false;

	for (dc_pipe_idx = 0;
			dc_pipe_idx < dc->res_pool->pipe_count; dc_pipe_idx++) {
		pipe = &context->res_ctx.pipe_ctx[dc_pipe_idx];
		if (resource_is_pipe_type(pipe, FREE_PIPE))
			continue;

		if (merge[dc_pipe_idx]) {
			if (resource_is_pipe_type(pipe, OPP_HEAD))
				/* merging OPP head means reducing ODM slice
				 * count by 1
				 */
				update_slice_table_for_stream(table, pipe->stream, -1);
			else if (resource_is_pipe_type(pipe, DPP_PIPE) &&
					resource_get_odm_slice_index(resource_get_opp_head(pipe)) == 0)
				/* merging DPP pipe of the first ODM slice means
				 * reducing MPC slice count by 1
				 */
				update_slice_table_for_plane(table, pipe, pipe->plane_state, -1);
			updated = true;
		}

		if (split[dc_pipe_idx]) {
			odm = vba->ODMCombineEnabled[vba->pipe_plane[dml_pipe_idx]] !=
					dm_odm_combine_mode_disabled;
			if (odm && resource_is_pipe_type(pipe, OPP_HEAD))
				update_slice_table_for_stream(
						table, pipe->stream, split[dc_pipe_idx] - 1);
			else if (!odm && resource_is_pipe_type(pipe, DPP_PIPE))
				update_slice_table_for_plane(table, pipe,
						pipe->plane_state, split[dc_pipe_idx] - 1);
			updated = true;
		}
		dml_pipe_idx++;
	}
	return updated;
}

static void update_pipes_with_slice_table(struct dc *dc, struct dc_state *context,
		struct pipe_slice_table *table)
{
	int i;

	for (i = 0; i < table->odm_combine_count; i++)
		resource_update_pipes_for_stream_with_slice_count(context,
				dc->current_state, dc->res_pool,
				table->odm_combines[i].stream,
				table->odm_combines[i].slice_count);

	for (i = 0; i < table->mpc_combine_count; i++)
		resource_update_pipes_for_plane_with_slice_count(context,
				dc->current_state, dc->res_pool,
				table->mpc_combines[i].plane,
				table->mpc_combines[i].slice_count);
}

static bool update_pipes_with_split_flags(struct dc *dc, struct dc_state *context,
		struct vba_vars_st *vba, int split[MAX_PIPES],
		bool merge[MAX_PIPES])
{
	struct pipe_slice_table slice_table;
	bool updated;

	init_pipe_slice_table_from_context(&slice_table, context);
	updated = update_pipe_slice_table_with_split_flags(
			&slice_table, dc, context, vba,
			split, merge);
	update_pipes_with_slice_table(dc, context, &slice_table);
	return updated;
}

static bool should_apply_odm_power_optimization(struct dc *dc,
		struct dc_state *context, struct vba_vars_st *v, int *split,
		bool *merge)
{
	struct dc_stream_state *stream = context->streams[0];
	struct pipe_slice_table slice_table;
	int i;

	/*
	 * this debug flag allows us to disable ODM power optimization feature
	 * unconditionally. we force the feature off if this is set to false.
	 */
	if (!dc->debug.enable_single_display_2to1_odm_policy)
		return false;

	/* current design and test coverage is only limited to allow ODM power
	 * optimization for single stream. Supporting it for multiple streams
	 * use case would require additional algorithm to decide how to
	 * optimize power consumption when there are not enough free pipes to
	 * allocate for all the streams. This level of optimization would
	 * require multiple attempts of revalidation to make an optimized
	 * decision. Unfortunately We do not support revalidation flow in
	 * current version of DML.
	 */
	if (context->stream_count != 1)
		return false;

	/*
	 * Our hardware doesn't support ODM for HDMI TMDS
	 */
	if (dc_is_hdmi_signal(stream->signal))
		return false;

	/*
	 * ODM Combine 2:1 requires horizontal timing divisible by 2 so each
	 * ODM segment has the same size.
	 */
	if (!is_h_timing_divisible_by_2(stream))
		return false;

	/*
	 * No power benefits if the timing's pixel clock is not high enough to
	 * raise display clock from minimum power state.
	 */
	if (stream->timing.pix_clk_100hz * 100 <= DCN3_2_VMIN_DISPCLK_HZ)
		return false;

	if (dc->config.enable_windowed_mpo_odm) {
		/*
		 * ODM power optimization should only be allowed if the feature
		 * can be seamlessly toggled off within an update. This would
		 * require that the feature is applied on top of a minimal
		 * state. A minimal state is defined as a state validated
		 * without the need of pipe split. Therefore, when transition to
		 * toggle the feature off, the same stream and plane
		 * configuration can be supported by the pipe resource in the
		 * first ODM slice alone without the need to acquire extra
		 * resources.
		 */
		init_pipe_slice_table_from_context(&slice_table, context);
		update_pipe_slice_table_with_split_flags(
				&slice_table, dc, context, v,
				split, merge);
		for (i = 0; i < slice_table.mpc_combine_count; i++)
			if (slice_table.mpc_combines[i].slice_count > 1)
				return false;

		for (i = 0; i < slice_table.odm_combine_count; i++)
			if (slice_table.odm_combines[i].slice_count > 1)
				return false;
	} else {
		/*
		 * the new ODM power optimization feature reduces software
		 * design limitation and allows ODM power optimization to be
		 * supported even with presence of overlay planes. The new
		 * feature is enabled based on enable_windowed_mpo_odm flag. If
		 * the flag is not set, we limit our feature scope due to
		 * previous software design limitation
		 */
		if (context->stream_status[0].plane_count != 1)
			return false;

		if (memcmp(&context->stream_status[0].plane_states[0]->clip_rect,
				&stream->src, sizeof(struct rect)) != 0)
			return false;

		if (stream->src.width >= 5120 &&
				stream->src.width > stream->dst.width)
			return false;
	}
	return true;
}

static void try_odm_power_optimization_and_revalidate(
		struct dc *dc,
		struct dc_state *context,
		display_e2e_pipe_params_st *pipes,
		int *split,
		bool *merge,
		unsigned int *vlevel,
		int pipe_cnt)
{
	int i;
	unsigned int new_vlevel;
	unsigned int cur_policy[MAX_PIPES];

	for (i = 0; i < pipe_cnt; i++) {
		cur_policy[i] = pipes[i].pipe.dest.odm_combine_policy;
		pipes[i].pipe.dest.odm_combine_policy = dm_odm_combine_policy_2to1;
	}

	new_vlevel = dml_get_voltage_level(&context->bw_ctx.dml, pipes, pipe_cnt);

	if (new_vlevel < context->bw_ctx.dml.soc.num_states) {
		memset(split, 0, MAX_PIPES * sizeof(int));
		memset(merge, 0, MAX_PIPES * sizeof(bool));
		*vlevel = dcn20_validate_apply_pipe_split_flags(dc, context, new_vlevel, split, merge);
		context->bw_ctx.dml.vba.VoltageLevel = *vlevel;
	} else {
		for (i = 0; i < pipe_cnt; i++)
			pipes[i].pipe.dest.odm_combine_policy = cur_policy[i];
	}
}

static bool is_test_pattern_enabled(
		struct dc_state *context)
{
	int i;

	for (i = 0; i < context->stream_count; i++) {
		if (context->streams[i]->test_pattern.type != DP_TEST_PATTERN_VIDEO_MODE)
			return true;
	}

	return false;
}

static bool dcn32_full_validate_bw_helper(struct dc *dc,
				   struct dc_state *context,
				   display_e2e_pipe_params_st *pipes,
				   int *vlevel,
				   int *split,
				   bool *merge,
				   int *pipe_cnt,
				   bool *repopulate_pipes)
{
	struct vba_vars_st *vba = &context->bw_ctx.dml.vba;
	unsigned int dc_pipe_idx = 0;
	int i = 0;
	bool found_supported_config = false;
	int vlevel_temp = 0;

	dc_assert_fp_enabled();

	/*
	 * DML favors voltage over p-state, but we're more interested in
	 * supporting p-state over voltage. We can't support p-state in
	 * prefetch mode > 0 so try capping the prefetch mode to start.
	 * Override present for testing.
	 */
	if (dc->debug.dml_disallow_alternate_prefetch_modes)
		context->bw_ctx.dml.soc.allow_for_pstate_or_stutter_in_vblank_final =
			dm_prefetch_support_uclk_fclk_and_stutter;
	else
		context->bw_ctx.dml.soc.allow_for_pstate_or_stutter_in_vblank_final =
			dm_prefetch_support_uclk_fclk_and_stutter_if_possible;

	*vlevel = dml_get_voltage_level(&context->bw_ctx.dml, pipes, *pipe_cnt);
	/* This may adjust vlevel and maxMpcComb */
	if (*vlevel < context->bw_ctx.dml.soc.num_states) {
		*vlevel = dcn20_validate_apply_pipe_split_flags(dc, context, *vlevel, split, merge);
		vba->VoltageLevel = *vlevel;
	}

	/* Apply split and merge flags before checking for subvp */
	if (!dcn32_apply_merge_split_flags_helper(dc, context, repopulate_pipes, split, merge))
		return false;
	memset(split, 0, MAX_PIPES * sizeof(int));
	memset(merge, 0, MAX_PIPES * sizeof(bool));

	/* Conditions for setting up phantom pipes for SubVP:
	 * 1. Not force disable SubVP
	 * 2. Full update (i.e. !fast_validate)
	 * 3. Enough pipes are available to support SubVP (TODO: Which pipes will use VACTIVE / VBLANK / SUBVP?)
	 * 4. Display configuration passes validation
	 * 5. (Config doesn't support MCLK in VACTIVE/VBLANK || dc->debug.force_subvp_mclk_switch)
	 */
	if (!dc->debug.force_disable_subvp && !dc->caps.dmub_caps.gecc_enable && dcn32_all_pipes_have_stream_and_plane(dc, context) &&
	    !dcn32_mpo_in_use(context) && !dcn32_any_surfaces_rotated(dc, context) && !is_test_pattern_enabled(context) &&
		(*vlevel == context->bw_ctx.dml.soc.num_states || (vba->DRAMSpeedPerState[*vlevel] != vba->DRAMSpeedPerState[0] &&
				vba->DRAMClockChangeSupport[*vlevel][vba->maxMpcComb] != dm_dram_clock_change_unsupported) ||
	    vba->DRAMClockChangeSupport[*vlevel][vba->maxMpcComb] == dm_dram_clock_change_unsupported ||
	    dc->debug.force_subvp_mclk_switch)) {

		vlevel_temp = *vlevel;

		while (!found_supported_config && dcn32_enough_pipes_for_subvp(dc, context) &&
			dcn32_assign_subvp_pipe(dc, context, &dc_pipe_idx)) {
			/* For the case where *vlevel = num_states, bandwidth validation has failed for this config.
			 * Adding phantom pipes won't change the validation result, so change the DML input param
			 * for P-State support before adding phantom pipes and recalculating the DML result.
			 * However, this case is only applicable for SubVP + DRR cases because the prefetch mode
			 * will not allow for switch in VBLANK. The DRR display must have it's VBLANK stretched
			 * enough to support MCLK switching.
			 */
			if (*vlevel == context->bw_ctx.dml.soc.num_states &&
				context->bw_ctx.dml.soc.allow_for_pstate_or_stutter_in_vblank_final ==
					dm_prefetch_support_uclk_fclk_and_stutter) {
				context->bw_ctx.dml.soc.allow_for_pstate_or_stutter_in_vblank_final =
								dm_prefetch_support_fclk_and_stutter;
				/* There are params (such as FabricClock) that need to be recalculated
				 * after validation fails (otherwise it will be 0). Calculation for
				 * phantom vactive requires call into DML, so we must ensure all the
				 * vba params are valid otherwise we'll get incorrect phantom vactive.
				 */
				*vlevel = dml_get_voltage_level(&context->bw_ctx.dml, pipes, *pipe_cnt);
			}

			dc->res_pool->funcs->add_phantom_pipes(dc, context, pipes, *pipe_cnt, dc_pipe_idx);

			*pipe_cnt = dc->res_pool->funcs->populate_dml_pipes(dc, context, pipes, false);
			// Populate dppclk to trigger a recalculate in dml_get_voltage_level
			// so the phantom pipe DLG params can be assigned correctly.
			pipes[0].clks_cfg.dppclk_mhz = get_dppclk_calculated(&context->bw_ctx.dml, pipes, *pipe_cnt, 0);
			*vlevel = dml_get_voltage_level(&context->bw_ctx.dml, pipes, *pipe_cnt);

			/* Check that vlevel requested supports pstate or not
			 * if not, select the lowest vlevel that supports it
			 */
			for (i = *vlevel; i < context->bw_ctx.dml.soc.num_states; i++) {
				if (vba->DRAMClockChangeSupport[i][vba->maxMpcComb] != dm_dram_clock_change_unsupported) {
					*vlevel = i;
					break;
				}
			}

			if (*vlevel < context->bw_ctx.dml.soc.num_states
			    && subvp_validate_static_schedulability(dc, context, *vlevel))
				found_supported_config = true;
			if (found_supported_config) {
				// For SubVP + DRR cases, we can force the lowest vlevel that supports the mode
				if (dcn32_subvp_drr_admissable(dc, context) && subvp_drr_schedulable(dc, context)) {
					/* find lowest vlevel that supports the config */
					for (i = *vlevel; i >= 0; i--) {
						if (vba->ModeSupport[i][vba->maxMpcComb]) {
							*vlevel = i;
						} else {
							break;
						}
					}
				}
			}
		}

		if (vba->DRAMSpeedPerState[*vlevel] >= vba->DRAMSpeedPerState[vlevel_temp])
			found_supported_config = false;

		// If SubVP pipe config is unsupported (or cannot be used for UCLK switching)
		// remove phantom pipes and repopulate dml pipes
		if (!found_supported_config) {
			dc_state_remove_phantom_streams_and_planes(dc, context);
			dc_state_release_phantom_streams_and_planes(dc, context);
			vba->DRAMClockChangeSupport[*vlevel][vba->maxMpcComb] = dm_dram_clock_change_unsupported;
			*pipe_cnt = dc->res_pool->funcs->populate_dml_pipes(dc, context, pipes, false);

			*vlevel = dml_get_voltage_level(&context->bw_ctx.dml, pipes, *pipe_cnt);
			/* This may adjust vlevel and maxMpcComb */
			if (*vlevel < context->bw_ctx.dml.soc.num_states) {
				*vlevel = dcn20_validate_apply_pipe_split_flags(dc, context, *vlevel, split, merge);
				vba->VoltageLevel = *vlevel;
			}
		} else {
			// Most populate phantom DLG params before programming hardware / timing for phantom pipe
			dcn32_helper_populate_phantom_dlg_params(dc, context, pipes, *pipe_cnt);

			/* Call validate_apply_pipe_split flags after calling DML getters for
			 * phantom dlg params, or some of the VBA params indicating pipe split
			 * can be overwritten by the getters.
			 *
			 * When setting up SubVP config, all pipes are merged before attempting to
			 * add phantom pipes. If pipe split (ODM / MPC) is required, both the main
			 * and phantom pipes will be split in the regular pipe splitting sequence.
			 */
			*vlevel = dcn20_validate_apply_pipe_split_flags(dc, context, *vlevel, split, merge);
			vba->VoltageLevel = *vlevel;
			// Note: We can't apply the phantom pipes to hardware at this time. We have to wait
			// until driver has acquired the DMCUB lock to do it safely.
			assign_subvp_index(dc, context);
		}
	}

	if (should_apply_odm_power_optimization(dc, context, vba, split, merge))
		try_odm_power_optimization_and_revalidate(
				dc, context, pipes, split, merge, vlevel, *pipe_cnt);

	return true;
}

static bool is_dtbclk_required(struct dc *dc, struct dc_state *context)
{
	int i;

	for (i = 0; i < dc->res_pool->pipe_count; i++) {
		if (!context->res_ctx.pipe_ctx[i].stream)
			continue;
		if (dc->link_srv->dp_is_128b_132b_signal(&context->res_ctx.pipe_ctx[i]))
			return true;
	}
	return false;
}

static void dcn20_adjust_freesync_v_startup(const struct dc_crtc_timing *dc_crtc_timing, int *vstartup_start)
{
	struct dc_crtc_timing patched_crtc_timing;
	uint32_t asic_blank_end   = 0;
	uint32_t asic_blank_start = 0;
	uint32_t newVstartup	  = 0;

	patched_crtc_timing = *dc_crtc_timing;

	if (patched_crtc_timing.flags.INTERLACE == 1) {
		if (patched_crtc_timing.v_front_porch < 2)
			patched_crtc_timing.v_front_porch = 2;
	} else {
		if (patched_crtc_timing.v_front_porch < 1)
			patched_crtc_timing.v_front_porch = 1;
	}

	/* blank_start = frame end - front porch */
	asic_blank_start = patched_crtc_timing.v_total -
					patched_crtc_timing.v_front_porch;

	/* blank_end = blank_start - active */
	asic_blank_end = asic_blank_start -
					patched_crtc_timing.v_border_bottom -
					patched_crtc_timing.v_addressable -
					patched_crtc_timing.v_border_top;

	newVstartup = asic_blank_end + (patched_crtc_timing.v_total - asic_blank_start);

	*vstartup_start = ((newVstartup > *vstartup_start) ? newVstartup : *vstartup_start);
}

static void dcn32_calculate_dlg_params(struct dc *dc, struct dc_state *context,
				       display_e2e_pipe_params_st *pipes,
				       int pipe_cnt, int vlevel)
{
	int i, pipe_idx, active_hubp_count = 0;
	bool usr_retraining_support = false;
	bool unbounded_req_enabled = false;
	struct vba_vars_st *vba = &context->bw_ctx.dml.vba;

	dc_assert_fp_enabled();

	/* Writeback MCIF_WB arbitration parameters */
	dc->res_pool->funcs->set_mcif_arb_params(dc, context, pipes, pipe_cnt);

	context->bw_ctx.bw.dcn.clk.dispclk_khz = context->bw_ctx.dml.vba.DISPCLK * 1000;
	context->bw_ctx.bw.dcn.clk.dcfclk_khz = context->bw_ctx.dml.vba.DCFCLK * 1000;
	context->bw_ctx.bw.dcn.clk.socclk_khz = context->bw_ctx.dml.vba.SOCCLK * 1000;
	context->bw_ctx.bw.dcn.clk.dramclk_khz = context->bw_ctx.dml.vba.DRAMSpeed * 1000 / 16;
	context->bw_ctx.bw.dcn.clk.dcfclk_deep_sleep_khz = context->bw_ctx.dml.vba.DCFCLKDeepSleep * 1000;
	context->bw_ctx.bw.dcn.clk.fclk_khz = context->bw_ctx.dml.vba.FabricClock * 1000;
	context->bw_ctx.bw.dcn.clk.p_state_change_support =
			context->bw_ctx.dml.vba.DRAMClockChangeSupport[vlevel][context->bw_ctx.dml.vba.maxMpcComb]
					!= dm_dram_clock_change_unsupported;

	/* Pstate change might not be supported by hardware, but it might be
	 * possible with firmware driven vertical blank stretching.
	 */
	context->bw_ctx.bw.dcn.clk.p_state_change_support |= context->bw_ctx.bw.dcn.clk.fw_based_mclk_switching;

	context->bw_ctx.bw.dcn.clk.dppclk_khz = 0;
	context->bw_ctx.bw.dcn.clk.dtbclk_en = is_dtbclk_required(dc, context);
	context->bw_ctx.bw.dcn.clk.ref_dtbclk_khz = context->bw_ctx.dml.vba.DTBCLKPerState[vlevel] * 1000;
	if (context->bw_ctx.dml.vba.FCLKChangeSupport[vlevel][context->bw_ctx.dml.vba.maxMpcComb] == dm_fclock_change_unsupported)
		context->bw_ctx.bw.dcn.clk.fclk_p_state_change_support = false;
	else
		context->bw_ctx.bw.dcn.clk.fclk_p_state_change_support = true;

	usr_retraining_support = conte