/* SPDX-License-Identifier: GPL-2.0-only */

#include <console/console.h>
#include <device/pci.h>
#include <device/pci_ids.h>
#include <fsp/util.h>

static const uint8_t fsp_tracehub_guid[16] = {
	0x09, 0x59, 0xb3, 0x5f, 0x1c, 0x5a, 0x31, 0x4a,
	0xad, 0xaf, 0x57, 0x7b, 0x54, 0x68, 0x26, 0x3f,
};

static void tracehub_read_resources(struct device *dev)
{
	const struct hob_resource *tracehub_info_hob;

	/* Read standard PCI resources. */
	pci_dev_read_resources(dev);

	/*
	 * Find the Trace Hub HOB generated by Intel FSP. If the Trace Hub
	 * is configured to save data in DRAM, FSP will generate this HOB.
	 * This HOB contains address and length of the memory region used
	 * by Trace Hub to save traces. Mark this memory region as reserved.
	 */
	tracehub_info_hob = fsp_find_resource_hob_by_guid(fsp_tracehub_guid);
	if (!tracehub_info_hob) {
		printk(BIOS_INFO, "Trace Hub HOB not found\n");
		return;
	}
	printk(BIOS_DEBUG, "Trace Hub HOB found: addr=0x%08llx length=0x%08llx\n",
			tracehub_info_hob->addr, tracehub_info_hob->length);
	reserved_ram_range(dev, 0, tracehub_info_hob->addr,
			   tracehub_info_hob->length);
}

static struct device_operations dev_ops = {
	.read_resources		= tracehub_read_resources,
	.set_resources		= pci_dev_set_resources,
	.enable_resources	= pci_dev_enable_resources,
	.ops_pci		= &pci_dev_ops_pci,
};

static const unsigned short pci_device_ids[] = {
	PCI_DID_INTEL_WCL_TRACEHUB,
	PCI_DID_INTEL_PTL_H_TRACEHUB,
	PCI_DID_INTEL_PTL_U_H_TRACEHUB,
	PCI_DID_INTEL_MTL_TRACEHUB,
	PCI_DID_INTEL_ARL_TRACEHUB,
	PCI_DID_INTEL_RPL_TRACEHUB,
	0
};

static const struct pci_driver tracehub_driver __pci_driver = {
	.ops		= &dev_ops,
	.vendor		= PCI_VID_INTEL,
	.devices	= pci_device_ids,
};
