// Copyright 2016 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package route

import (
	"reflect"
	"syscall"
	"testing"
)

type parseAddrsOnDarwinTest struct {
	attrs uint
	fn    func(int, []byte) (int, Addr, error)
	b     []byte
	as    []Addr
}

var parseAddrsOnDarwinLittleEndianTests = []parseAddrsOnDarwinTest{
	{
		syscall.RTA_DST | syscall.RTA_GATEWAY | syscall.RTA_NETMASK,
		parseKernelInetAddr,
		[]byte{
			0x10, 0x2, 0x0, 0x0, 0xc0, 0xa8, 0x56, 0x0,
			0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,

			0x14, 0x12, 0x4, 0x0, 0x6, 0x0, 0x0, 0x0,
			0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
			0x0, 0x0, 0x0, 0x0,

			0x7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0,
		},
		[]Addr{
			&Inet4Addr{IP: [4]byte{192, 168, 86, 0}},
			&LinkAddr{Index: 4},
			&Inet4Addr{IP: [4]byte{255, 255, 255, 0}},
			nil,
			nil,
			nil,
			nil,
			nil,
		},
	},
	{
		syscall.RTA_DST | syscall.RTA_GATEWAY | syscall.RTA_NETMASK,
		parseKernelInetAddr,
		[]byte{
			0x10, 0x02, 0x00, 0x00, 0x64, 0x71, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

			0x14, 0x12, 0x21, 0x00, 0x01, 0x08, 0x00, 0x00,
			0x75, 0x74, 0x75, 0x6e, 0x34, 0x33, 0x31, 0x39,
			0x00, 0x00, 0x00, 0x00,

			0x06, 0x02, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
		},
		[]Addr{
			&Inet4Addr{IP: [4]byte{100, 113, 0, 0}},
			&LinkAddr{Index: 33, Name: "utun4319"},
			&Inet4Addr{IP: [4]byte{255, 255, 0, 0}},
			nil,
			nil,
			nil,
			nil,
			nil,
		},
	},
	// route -n add -inet6 fd84:1b4e:6281:: -prefixlen 48 fe80::f22f:4bff:fe09:3bff%utun4319
	// gw fe80:0000:0000:0000:f22f:4bff:fe09:3bff
	{
		syscall.RTA_DST | syscall.RTA_GATEWAY | syscall.RTA_NETMASK,
		parseKernelInetAddr,
		[]byte{
			0x1c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xfd, 0x84, 0x1b, 0x4e, 0x62, 0x81, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00,

			0x1c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xfe, 0x80, 0x00, 0x21, 0x00, 0x00, 0x00, 0x00,
			0xf2, 0x2f, 0x4b, 0xff, 0xfe, 0x09, 0x3b, 0xff,
			0x00, 0x00, 0x00, 0x00,

			0x0e, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
		},
		[]Addr{
			&Inet6Addr{IP: [16]byte{0xfd, 0x84, 0x1b, 0x4e, 0x62, 0x81}},
			&Inet6Addr{IP: [16]byte{0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0x2f, 0x4b, 0xff, 0xfe, 0x09, 0x3b, 0xff}, ZoneID: 33},
			&Inet6Addr{IP: [16]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff}},
			nil,
			nil,
			nil,
			nil,
			nil,
		},
	},
	// sudo route -n add -inet6 -ifscope en11 -net :: -netmask :: fe80::2d0:4cff:fe10:15d2
	// RTM_ADD: Add Route: len 152, pid: 81198, seq 1, errno 0, ifscope 13, flags:<UP,GATEWAY,DONE,STATIC,IFSCOPE>
	// locks:  inits:
	// sockaddrs: <DST,GATEWAY,NETMASK>
	// :: fe80::2d0:4cff:fe10:15d2 ::
	{
		syscall.RTA_DST | syscall.RTA_GATEWAY | syscall.RTA_NETMASK,
		parseKernelInetAddr,
		[]byte{
			0x1c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00,

			0x1c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x02, 0xd0, 0x4c, 0xff, 0xfe, 0x10, 0x15, 0xd2,
			0x00, 0x00, 0x00, 0x00,

			0x02, 0x1e, 0x00, 0x00,
		},
		[]Addr{
			&Inet6Addr{},
			&Inet6Addr{IP: [16]byte{0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xd0, 0x4c, 0xff, 0xfe, 0x10, 0x15, 0xd2}},
			&Inet6Addr{},
			nil,
			nil,
			nil,
			nil,
			nil,
		},
	},
	// golang/go#70528, the kernel can produce addresses of length 0
	{
		syscall.RTA_DST | syscall.RTA_GATEWAY | syscall.RTA_NETMASK,
		parseKernelInetAddr,
		[]byte{
			0x00, 0x1e, 0x00, 0x00,

			0x1c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xfe, 0x80, 0x00, 0x21, 0x00, 0x00, 0x00, 0x00,
			0xf2, 0x2f, 0x4b, 0xff, 0xfe, 0x09, 0x3b, 0xff,
			0x00, 0x00, 0x00, 0x00,

			0x0e, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
		},
		[]Addr{
			&Inet6Addr{IP: [16]byte{}},
			&Inet6Addr{IP: [16]byte{0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0x2f, 0x4b, 0xff, 0xfe, 0x09, 0x3b, 0xff}, ZoneID: 33},
			&Inet6Addr{IP: [16]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff}},
			nil,
			nil,
			nil,
			nil,
			nil,
		},
	},
	// Additional case: golang/go/issues/70528#issuecomment-2498692877
	{
		syscall.RTA_DST | syscall.RTA_GATEWAY | syscall.RTA_NETMASK,
		parseKernelInetAddr,
		[]byte{
			0x84, 0x00, 0x05, 0x04, 0x01, 0x00, 0x00, 0x00, 0x03, 0x08, 0x00, 0x01, 0x15, 0x00, 0x00, 0x00,
			0x1B, 0x01, 0x00, 0x00, 0xF5, 0x5A, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x02, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
			0x14, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00,
		},
		[]Addr{
			&Inet4Addr{IP: [4]byte{0x0, 0x0, 0x0, 0x0}},
			nil,
			nil,
			nil,
			nil,
			nil,
			nil,
			nil,
		},
	},
}

func TestParseAddrsOnDarwin(t *testing.T) {
	tests := parseAddrsOnDarwinLittleEndianTests
	if nativeEndian != littleEndian {
		t.Skip("no test for non-little endian machine yet")
	}

	for i, tt := range tests {
		as, err := parseAddrs(tt.attrs, tt.fn, tt.b)
		if err != nil {
			t.Error(i, err)
			continue
		}
		if !reflect.DeepEqual(as, tt.as) {
			t.Errorf("#%d: got %+v; want %+v", i, as, tt.as)
			continue
		}
	}
}
