import pytest

from libqtile import config, layout
from libqtile.confreader import Config
from libqtile.lazy import lazy


# Config with multiple keys and when checks
class WhenConfig(Config):
    keys = [
        config.Key(["control"], "k", lazy.window.toggle_floating()),
        config.Key(["control"], "p", lazy.window.toggle_floating().when(when_floating=True)),
        config.Key(["control"], "o", lazy.window.toggle_floating().when(when_floating=False)),
        config.Key(
            ["control"],
            "j",
            lazy.window.toggle_floating().when(focused=config.Match(wm_class="TestWindow")),
        ),
        config.Key(
            ["control"],
            "h",
            lazy.window.toggle_floating().when(focused=config.Match(wm_class="idonotexist")),
        ),
        config.Key(
            ["control"],
            "n",
            lazy.next_layout().when(focused=config.Match(wm_class="TestWindow")),
        ),
        config.Key(
            ["control"],
            "m",
            lazy.next_layout().when(
                focused=config.Match(wm_class="TestWindow"), if_no_focused=True
            ),
        ),
        config.Key(["control"], "t", lazy.next_layout().when(condition=1 + 1 == 2)),
        config.Key(["control"], "f", lazy.next_layout().when(condition=1 + 1 == 3)),
        config.Key(["control", "shift"], "t", lazy.next_layout().when(func=lambda: True)),
        config.Key(["control", "shift"], "f", lazy.next_layout().when(func=lambda: False)),
    ]
    layouts = [layout.MonadWide(), layout.MonadTall()]


when_config = pytest.mark.parametrize("manager", [WhenConfig], indirect=True)


@when_config
def test_when(manager):
    # Check if the test window is alive and tiled
    one = manager.test_window("one")
    assert not manager.c.window.info()["floating"]

    # This sets the window to floating as there is no when
    manager.c.simulate_keypress(["control"], "k")
    assert manager.c.window.info()["floating"]

    # This keeps the window floating as the class doesn't match
    manager.c.simulate_keypress(["control"], "h")
    assert manager.c.window.info()["floating"]

    # This sets the window tiled as the class does match
    manager.c.simulate_keypress(["control"], "j")
    assert not manager.c.window.info()["floating"]

    # This keeps the window tiled as window is not floating
    manager.c.simulate_keypress(["control"], "p")
    assert not manager.c.window.info()["floating"]

    # This sets the window floating as window is not floating
    manager.c.simulate_keypress(["control"], "o")
    assert manager.c.window.info()["floating"]

    # Kill the window to create an empty group
    manager.kill_window(one)
    prev_layout_info = manager.c.layout.info()

    # This does not go to the next layout as empty is not matched
    manager.c.simulate_keypress(["control"], "n")
    assert manager.c.layout.info() == prev_layout_info

    # This does go to the next layout as empty is matched
    manager.c.simulate_keypress(["control"], "m")
    assert manager.c.layout.info() != prev_layout_info

    # Test boolean argument
    prev_layout_info = manager.c.layout.info()

    manager.c.simulate_keypress(["control"], "f")
    assert manager.c.layout.info() == prev_layout_info

    manager.c.simulate_keypress(["control"], "t")
    assert manager.c.layout.info() != prev_layout_info

    # Test function argument
    prev_layout_info = manager.c.layout.info()

    manager.c.simulate_keypress(["control", "shift"], "f")
    assert manager.c.layout.info() == prev_layout_info

    manager.c.simulate_keypress(["control", "shift"], "t")
    assert manager.c.layout.info() != prev_layout_info
